import 'dart:convert';
import 'dart:io';
import 'dart:typed_data';
import 'package:flutter/material.dart';
import 'package:flutter_local_notifications/flutter_local_notifications.dart';
import 'package:firebase_messaging/firebase_messaging.dart';
import 'package:timezone/data/latest_all.dart' as tz;
import 'package:timezone/timezone.dart' as tz;
import 'package:http/http.dart' as http;

class NotificationService {
  static final FlutterLocalNotificationsPlugin _notifications =
  FlutterLocalNotificationsPlugin();

  static Function(Map<String, dynamic>)? onNotificationClick;

  static Future<void> init() async {
    tz.initializeTimeZones();

    // إعدادات Android
    const androidSettings = AndroidInitializationSettings('@mipmap/ic_launcher');

    // إعدادات iOS
    final iosSettings = DarwinInitializationSettings(
      requestAlertPermission: true,
      requestBadgePermission: true,
      requestSoundPermission: true,
    );

    final settings = InitializationSettings(
      android: androidSettings,
      iOS: iosSettings,
    );

    await _notifications.initialize(
      settings,
      onDidReceiveNotificationResponse: (NotificationResponse response) {
        if (response.payload != null) {
          try {
            final data = jsonDecode(response.payload!);
            if (onNotificationClick != null) onNotificationClick!(data);
          } catch (_) {}
        }
      },
    );

    if (Platform.isIOS) {
      await _notifications
          .resolvePlatformSpecificImplementation<IOSFlutterLocalNotificationsPlugin>()
          ?.requestPermissions(alert: true, badge: true, sound: true);
    }

    // FCM foreground listener
    FirebaseMessaging.onMessage.listen((message) async {
      String? imageUrl = message.data['image'] ?? message.notification?.android?.imageUrl?.toString() ?? message.notification?.apple?.imageUrl?.toString();
      await showNotification(
        message.notification?.title ?? '',
        message.notification?.body ?? '',
        imageUrl: imageUrl,
        payload: message.data,
      );
    });

    // FCM background click
    FirebaseMessaging.onMessageOpenedApp.listen((message) {
      if (onNotificationClick != null) onNotificationClick!(message.data);
    });
  }

  static Future<void> showNotification(
      String title, String body,
      {String? imageUrl, Map<String, dynamic>? payload}) async {
    BigPictureStyleInformation? bigPicture;

    if (imageUrl != null && imageUrl.isNotEmpty) {
      final bitmap = await _getAndroidBitmapFromUrl(imageUrl);
      if (bitmap != null) {
        bigPicture = BigPictureStyleInformation(bitmap);
      }
    }

    await _notifications.show(
      title.hashCode,
      title,
      body,
      NotificationDetails(
        android: AndroidNotificationDetails(
          'default_channel',
          'General',
          channelDescription: 'General notifications',
          importance: Importance.max,
          priority: Priority.high,
          styleInformation: bigPicture,
        ),
        iOS: const DarwinNotificationDetails(),
      ),
      payload: payload != null ? jsonEncode(payload) : null,
    );
  }

  static Future<ByteArrayAndroidBitmap?> _getAndroidBitmapFromUrl(String url) async {
    try {
      final response = await http.get(Uri.parse(url));
      if (response.statusCode == 200) {
        return ByteArrayAndroidBitmap(response.bodyBytes);
      }
      return null;
    } catch (_) {
      return null;
    }
  }

  static Future<void> scheduleNotification(
      String title, DateTime scheduledTime,
      {Map<String, dynamic>? payload}) async {
    await _notifications.zonedSchedule(
      title.hashCode,
      title,
      title,
      tz.TZDateTime.from(scheduledTime, tz.local),
      NotificationDetails(
        android: AndroidNotificationDetails(
          'remind_channel',
          'Reminders',
          channelDescription: 'Reminder notifications',
          importance: Importance.max,
          priority: Priority.high,
        ),
        iOS: const DarwinNotificationDetails(),
      ),
      androidScheduleMode: AndroidScheduleMode.exactAllowWhileIdle,
      payload: payload != null ? jsonEncode(payload) : null,
    );
  }
}
