import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import 'package:youtube_player_flutter/youtube_player_flutter.dart';
import '../models/exercise_model.dart';
import '../providers/locale_provider.dart';

class StartWorkoutScreen extends StatefulWidget {
  final Exercise exercise;

  const StartWorkoutScreen({super.key, required this.exercise});

  @override
  State<StartWorkoutScreen> createState() => _StartWorkoutScreenState();
}

class _StartWorkoutScreenState extends State<StartWorkoutScreen> {
  late YoutubePlayerController _youtubeController;
  bool _isYoutubeLink = false;
  String? _error;

  @override
  void initState() {
    super.initState();
    _initializePlayer();
  }

  void _initializePlayer() {
    final videoUrl = widget.exercise.video ?? "";

    if (videoUrl.isEmpty) {
      setState(() {
        _error = "لا يوجد فيديو لهذا التمرين";
      });
      return;
    }

    // تحقق إذا كان الرابط يوتيوب
    final videoId = YoutubePlayer.convertUrlToId(videoUrl);
    if (videoId == null) {
      setState(() {
        _error = "الرابط غير صالح أو غير مدعوم";
      });
      return;
    }

    _isYoutubeLink = true;
    _youtubeController = YoutubePlayerController(
      initialVideoId: videoId,
      flags: const YoutubePlayerFlags(
        autoPlay: true,
        mute: false,
        enableCaption: true,
        isLive: true, // للبث المباشر
      ),
    );
  }

  @override
  void dispose() {
    if (_isYoutubeLink) {
      _youtubeController.dispose();
    }
    super.dispose();
  }

  @override
  @override
  Widget build(BuildContext context) {
    final locale = Provider.of<LocaleProvider>(context).locale.languageCode;

    final title = locale == 'ar'
        ? (widget.exercise.title_ar ?? "")
        : (widget.exercise.title_en ?? "");

    final description = locale == 'ar'
        ? (widget.exercise.description_ar ?? "")
        : (widget.exercise.description_en ?? "");

    return Scaffold(
      backgroundColor: Colors.black,
      appBar: AppBar(
        title: Text(title),
        backgroundColor: Colors.black,
      ),
      body: _error != null
          ? Center(
        child: Column(
          mainAxisAlignment: MainAxisAlignment.center,
          children: [
            Text(
              _error!,
              style: const TextStyle(color: Colors.white),
              textAlign: TextAlign.center,
            ),
            const SizedBox(height: 12),
            ElevatedButton(
              onPressed: _initializePlayer,
              style: ElevatedButton.styleFrom(
                backgroundColor: Colors.red,
              ),
              child: const Text("إعادة المحاولة"),
            ),
          ],
        ),
      )
          : _isYoutubeLink
          ? Column(
        children: [
          // هنا نستخدم Expanded ليأخذ كل المساحة الممكنة للفيديو
          Expanded(
            child: YoutubePlayer(
              controller: _youtubeController,
              showVideoProgressIndicator: true,
              progressIndicatorColor: Colors.red,
            ),
          ),
          if (description.isNotEmpty)
            Padding(
              padding: const EdgeInsets.all(12.0),
              child: Text(
                description,
                style: const TextStyle(
                    color: Colors.white70, fontSize: 16),
                textAlign: TextAlign.center,
              ),
            ),
        ],
      )
          : const Center(
        child: CircularProgressIndicator(color: Colors.white),
      ),
    );
  }
}
