import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:hockey_app/l10n/app_localizations.dart';
import 'package:hockey_app/services/auth_service.dart';
import 'package:hockey_app/main.dart';
import 'home_screen.dart';
import 'onboarding_screen.dart';

class SignUpPage extends StatefulWidget {
  const SignUpPage({super.key});

  @override
  State<SignUpPage> createState() => _SignUpPageState();
}

class _SignUpPageState extends State<SignUpPage> {
  final _formKey = GlobalKey<FormState>();

  final TextEditingController _firstNameController = TextEditingController();
  final TextEditingController _lastNameController = TextEditingController();
  final TextEditingController _emailController = TextEditingController();
  final TextEditingController _passwordController = TextEditingController();
  final TextEditingController _phoneController = TextEditingController();

  bool receiveUpdates = false;
  bool isLoading = false;

  String? selectedRole;
  final List<String> roles = ["Player", "Coach", "Guest"];

  void _submitForm() async {
    final t = AppLocalizations.of(context)!;

    if (_formKey.currentState!.validate()) {
      setState(() => isLoading = true);

      final data = {
        "first_name": _firstNameController.text.trim(),
        "last_name": _lastNameController.text.trim(),
        "email": _emailController.text.trim(),
        "password": _passwordController.text.trim(),
        "phone": _phoneController.text.trim(),
        "role": selectedRole,
      };

      try {
        final result = await AuthService.signup(data);
        print(result); // <-- هذا سيطبع الرد من السيرفر

        if (result["message"] == "User registered successfully") {
          await saveUserSession(result["token"], result['user']['id']);
          Navigator.pushReplacement(
            context,
            MaterialPageRoute(builder: (context) => MainScreen()),
          );
        }
      } catch (e) {
        String errorMessage = e.toString();

        // ترجمة الأخطاء الشائعة
        if (errorMessage.contains("Email already exists")) {
          errorMessage = t.emailAlreadyExists; // ضع هذا في ملفات اللغات
        } else if (errorMessage.contains("Invalid password")) {
          errorMessage = t.invalid_email_or_password;
        }
        _showDialog(t.signup, errorMessage);
        print(e);
      } finally {
        setState(() => isLoading = false);
      }
    }
  }

  void _showDialog(String title, String message) {
    showDialog(
      context: context,
      builder: (_) => AlertDialog(
        shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(20)),
        title: Text(title, style: const TextStyle(fontWeight: FontWeight.bold)),
        content: Text(message),
        actions: [
          TextButton(
            onPressed: () => Navigator.of(context).pop(),
            child: const Text("OK"),
          ),
        ],
      ),
    );
  }

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    final textTheme = theme.textTheme;
    final t = AppLocalizations.of(context)!;

    return Scaffold(
      backgroundColor: theme.scaffoldBackgroundColor,
      appBar: AppBar(
        backgroundColor: theme.appBarTheme.backgroundColor,
        leading: BackButton(
          color: theme.appBarTheme.foregroundColor,
          onPressed: () => Navigator.pushAndRemoveUntil(
            context,
            MaterialPageRoute(builder: (context) => const OnboardingScreen()),
            (Route<dynamic> route) => false,
          ),
        ),
        title: Text(t.welcome, style: textTheme.titleLarge),
        centerTitle: true,
      ),
      body: Padding(
        padding: const EdgeInsets.all(24),
        child: Form(
          key: _formKey,
          child: ListView(
            children: [
              _buildTextField(_firstNameController, Icons.person, t.firstName,
                  validator: (val) =>
                      val == null || val.isEmpty ? t.enterFirstName : null),
              const SizedBox(height: 16),
              _buildTextField(_lastNameController, Icons.person, t.lastName,
                  validator: (val) =>
                      val == null || val.isEmpty ? t.enterLastName : null),
              const SizedBox(height: 16),
              _buildTextField(_emailController, Icons.email, t.emailAddress,
                  validator: (val) {
                if (val == null || val.isEmpty) return t.enterEmail;
                if (!RegExp(r'^[^@]+@[^@]+\.[^@]+').hasMatch(val)) {
                  return t.enterValidEmail;
                }
                return null;
              }),
              const SizedBox(height: 16),
              _buildTextField(_passwordController, Icons.lock, t.password,
                  obscure: true, validator: (val) {
                if (val == null || val.isEmpty) return t.enterPassword;
                if (val.length < 8) return t.passwordMin8Chars;
                return null;
              }),
              const SizedBox(height: 8),
              Text(
                t.passwordInfo,
                style: textTheme.labelLarge?.copyWith(color: Colors.grey),
              ),
              const SizedBox(height: 16),
              _buildTextField(_phoneController, Icons.phone, t.phoneOptional,
                  validator: (val) {
                if (val != null && val.isNotEmpty) {
                  if (!RegExp(r'^[0-9]+$').hasMatch(val)) {
                    return t.phoneDigitsOnly;
                  }
                }
                return null;
              }),
              const SizedBox(height: 16),
              DropdownButtonFormField<String>(
                value: selectedRole,
                items: roles
                    .map((role) => DropdownMenuItem(
                          value: role,
                          child: Text(role),
                        ))
                    .toList(),
                onChanged: (value) {
                  setState(() {
                    selectedRole = value;
                  });
                },
                decoration: InputDecoration(
                  prefixIcon: const Icon(Icons.group),
                  hintText: t.selectRole,
                ),
                validator: (val) => val == null ? t.selectRoleValidation : null,
              ),
              const SizedBox(height: 20),
              Text(
                t.privacyPolicyNotice,
                style: textTheme.labelLarge?.copyWith(color: Colors.grey),
              ),
              const SizedBox(height: 20),
              ElevatedButton(
                onPressed: isLoading ? null : _submitForm,
                style: theme.elevatedButtonTheme.style?.copyWith(
                  minimumSize: MaterialStateProperty.all(
                      const Size(double.infinity, 55)),
                ),
                child: isLoading
                    ? const CircularProgressIndicator(color: Colors.white)
                    : Text(t.next),
              ),
            ],
          ),
        ),
      ),
    );
  }

  Widget _buildTextField(
      TextEditingController controller, IconData icon, String hint,
      {bool obscure = false, String? Function(String?)? validator}) {
    return TextFormField(
      controller: controller,
      obscureText: obscure,
      decoration: InputDecoration(
        prefixIcon: Icon(icon),
        hintText: hint,
      ),
      validator: validator,
    );
  }
}
