import 'package:flutter/material.dart';
import 'package:hockey_app/screens/login.dart';
import 'package:intl/intl.dart' show DateFormat;
import 'package:provider/provider.dart';
import 'package:shared_preferences/shared_preferences.dart';
import '../providers/locale_provider.dart';
import '../providers/theme_provider.dart';
import '../services/user_service.dart';
import '../models/user_model.dart';
import '../screens/reset_password.dart';
import '../l10n/app_localizations.dart';

class SettingsScreen extends StatefulWidget {
  const SettingsScreen({super.key});

  @override
  State<SettingsScreen> createState() => _SettingsScreenState();
}

class _SettingsScreenState extends State<SettingsScreen> {
  User? user;
  bool isLoading = true;

  final _formKey = GlobalKey<FormState>();

  late TextEditingController firstNameController;
  late TextEditingController lastNameController;
  late TextEditingController emailController;
  late TextEditingController phoneController;
  late TextEditingController heightController;
  late TextEditingController weightController;
  String? gender;
  DateTime? birthday;
  String? imageUrl;

  @override
  void initState() {
    super.initState();
    _loadUser();
  }

  Future<void> _loadUser() async {
    try {
      final fetchedUser = await UserService.getUser();
      setState(() {
        user = fetchedUser;
        firstNameController = TextEditingController(text: user?.first_name ?? '');
        lastNameController = TextEditingController(text: user?.last_name ?? '');
        emailController = TextEditingController(text: user?.email ?? '');
        phoneController = TextEditingController(text: user?.phone ?? '');
        heightController = TextEditingController(text: user?.height?.toString() ?? '');
        weightController = TextEditingController(text: user?.weight?.toString() ?? '');
        gender = user?.gender ?? 'Male';
        birthday = user?.birthDate;
        imageUrl = user?.profileImageUrl;
        isLoading = false;
      });
    } catch (_) {
      setState(() => isLoading = false);
    }
  }

  static Future<void> logout() async {
    final prefs = await SharedPreferences.getInstance();
    await prefs.remove("auth_token");
    await prefs.remove("user_id");
    await prefs.clear();
  }

  Future<void> _saveProfile() async {
    if (_formKey.currentState!.validate()) {
      final updatedUser = User(
        first_name: firstNameController.text,
        last_name: lastNameController.text,
        email: emailController.text,
        phone: phoneController.text,
        height: heightController.text,
        weight: weightController.text,
        gender: gender,
        birthDate: birthday,
        profileImageUrl: imageUrl,
      );

      await UserService.updateUser(updatedUser);

      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text(AppLocalizations.of(context)!.profile_updated)),
      );
    }
  }

  @override
  Widget build(BuildContext context) {
    final localProvider = Provider.of<LocaleProvider>(context);
    final themeProvider = Provider.of<ThemeProvider>(context);
    final tr = AppLocalizations.of(context)!;

    if (isLoading) {
      return const Scaffold(
        body: Center(child: CircularProgressIndicator()),
      );
    }

    if (user == null) {
      return Scaffold(
        body: Center(child: Text(tr.loading_error)),
      );
    }

    return Scaffold(
      appBar: AppBar(
        title: Text(tr.profile),
        actions: [
          IconButton(
            icon: const Icon(Icons.save),
            onPressed: _saveProfile,
          )
        ],
      ),
      body: ListView(
        padding: const EdgeInsets.all(16),
        children: [
          const SizedBox(height: 16),
          Form(
            key: _formKey,
            child: Column(
              children: [
                // الاسم الأول والاخير
                TextFormField(
                  controller: firstNameController,
                  decoration: InputDecoration(
                    labelText: tr.first_name,
                    prefixIcon: const Icon(Icons.person),
                  ),
                  validator: (value) => value!.isEmpty ? tr.required_field : null,
                ),
                const SizedBox(height: 12),
                TextFormField(
                  controller: lastNameController,
                  decoration: InputDecoration(
                    labelText: tr.last_name,
                    prefixIcon: const Icon(Icons.person),
                  ),
                  validator: (value) => value!.isEmpty ? tr.required_field : null,
                ),
                const SizedBox(height: 12),

                // الجنس
                DropdownButtonFormField<String>(
                  value: (gender == 'Male' || gender == 'Female') ? gender : null,
                  decoration: InputDecoration(
                    labelText: tr.gender,
                    prefixIcon: const Icon(Icons.wc),
                  ),
                  items: [
                    DropdownMenuItem(value: 'Male', child: Text(tr.male)),
                    DropdownMenuItem(value: 'Female', child: Text(tr.female)),
                  ],
                  onChanged: (val) => setState(() => gender = val),
                  validator: (val) => val == null ? tr.required_field : null,
                ),
                const SizedBox(height: 12),

                // تاريخ الميلاد
                TextFormField(
                  readOnly: true,
                  decoration: InputDecoration(
                    labelText: tr.birthDate,
                    prefixIcon: const Icon(Icons.cake),
                  ),
                  controller: TextEditingController(
                      text: birthday != null
                          ? DateFormat('yyyy-MM-dd').format(birthday!)
                          : ''),
                  onTap: () async {
                    final picked = await showDatePicker(
                      context: context,
                      initialDate: birthday ?? DateTime(2000, 1, 1),
                      firstDate: DateTime(1900),
                      lastDate: DateTime.now(),
                    );
                    if (picked != null) setState(() => birthday = picked);
                  },
                ),
                const SizedBox(height: 12),

                // البريد والهاتف والطول والوزن
                TextFormField(
                  controller: emailController,
                  decoration: InputDecoration(
                    labelText: tr.email,
                    prefixIcon: const Icon(Icons.email),
                  ),
                  validator: (value) => value!.isEmpty ? tr.required_field : null,
                ),
                const SizedBox(height: 12),
                TextFormField(
                  controller: phoneController,
                  decoration: InputDecoration(
                    labelText: tr.phone,
                    prefixIcon: const Icon(Icons.phone),
                  ),
                  validator: (value) => value!.isEmpty ? tr.required_field : null,
                ),
                const SizedBox(height: 12),
                TextFormField(
                  controller: heightController,
                  decoration: InputDecoration(
                    labelText: tr.height,
                    prefixIcon: const Icon(Icons.height),
                  ),
                  keyboardType: TextInputType.number,
                ),
                const SizedBox(height: 12),
                TextFormField(
                  controller: weightController,
                  decoration: InputDecoration(
                    labelText: tr.weight,
                    prefixIcon: const Icon(Icons.fitness_center),
                  ),
                  keyboardType: TextInputType.number,
                ),
                const SizedBox(height: 24),

                // الإعدادات
                _buildSettingsTile(
                  context,
                  icon: Icons.language,
                  title: tr.language,
                  subtitle: localProvider.locale.languageCode == 'en'
                      ? tr.english
                      : tr.arabic,
                  onTap: () async {
                    final selected = await showDialog<String>(
                      context: context,
                      builder: (context) => SimpleDialog(
                        title: Text(tr.language),
                        children: [
                          SimpleDialogOption(
                              onPressed: () => Navigator.pop(context, 'en'),
                              child: Text(tr.english)),
                          SimpleDialogOption(
                              onPressed: () => Navigator.pop(context, 'ar'),
                              child: Text(tr.arabic)),
                        ],
                      ),
                    );
                    if (selected != null) localProvider.setLocale(selected);
                  },
                ),
                _buildSettingsTile(
                  context,
                  icon: Icons.dark_mode,
                  title: tr.theme,
                  subtitle: themeProvider.isDarkMode ? tr.dark : tr.light,
                  onTap: () => themeProvider.toggleTheme(!themeProvider.isDarkMode),
                ),
                _buildSettingsTile(
                  context,
                  icon: Icons.lock_outline,
                  title: tr.change_password,
                  onTap: () => Navigator.push(
                      context,
                      MaterialPageRoute(builder: (_) => const ResetPasswordScreen())),
                ),
                _buildSettingsTile(
                  context,
                  icon: Icons.exit_to_app,
                  title: tr.logout,
                  onTap: () async {
                    await logout();
                    if (mounted) {
                      Navigator.pushAndRemoveUntil(
                        context,
                        MaterialPageRoute(builder: (_) => const LoginPage()),
                            (route) => false,
                      );
                    }
                  },
                ),
              ],
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildSettingsTile(BuildContext context,
      {required IconData icon,
        required String title,
        String? subtitle,
        required VoidCallback onTap}) {
    return ListTile(
      leading: Icon(icon, color: Theme.of(context).colorScheme.primary),
      title: Text(title),
      subtitle: subtitle != null ? Text(subtitle) : null,
      trailing: const Icon(Icons.arrow_forward_ios, size: 16),
      onTap: onTap,
    );
  }
}
