import 'dart:async';
import 'package:flutter/material.dart';
import 'package:hockey_app/l10n/app_localizations.dart';
import 'package:hockey_app/services/api_constants.dart';
import 'package:provider/provider.dart';
import 'package:video_player/video_player.dart';
import 'package:hockey_app/screens/login.dart';
import 'package:hockey_app/screens/sign_up_page.dart';
import 'package:hockey_app/services/onboarding_service.dart';
import 'package:hockey_app/models/on_boarding_model.dart';

import '../providers/locale_provider.dart';

class OnboardingScreen extends StatefulWidget {
  const OnboardingScreen({super.key});

  @override
  State<OnboardingScreen> createState() => _OnboardingScreenState();
}

class _OnboardingScreenState extends State<OnboardingScreen> {
  final PageController _pageController = PageController();
  int _currentIndex = 0;
  VideoPlayerController? _videoController;
  Timer? _timer;

  OnboardingModel? onboarding; // البيانات من الـ API
  bool loading = true;

  @override
  void initState() {
    super.initState();
    _loadOnboarding();
  }

  Future<void> _loadOnboarding() async {
    try {
      final data = await OnboardingService().fetchOnboarding();
      setState(() {
        onboarding = data;
        loading = false;
      });

      if (onboarding!.type == "video" && onboarding!.video != null) {
        _initVideo(onboarding!.video!);
      } else {
        _startAutoSlide();
      }
    } catch (e) {
      debugPrint("Error loading onboarding: $e");
    }
  }

  void _initVideo(String url) {
    _videoController = VideoPlayerController.networkUrl(
      Uri.parse(ApiConstants.baseUrl+'/uploads/videos/'+url)
    )..initialize().then((_) {
        _videoController!.setLooping(true);
        _videoController!.setVolume(0);
        _videoController!.play();
        setState(() {});
      });
  }

  void _startAutoSlide() {
    _timer = Timer.periodic(const Duration(seconds: 4), (timer) {
      if (!mounted) return;
      if (_currentIndex < 2) {
        _currentIndex++;
      } else {
        _currentIndex = 0;
      }
      _pageController.animateToPage(
        _currentIndex,
        duration: const Duration(milliseconds: 700),
        curve: Curves.easeInOut,
      );
    });
  }

  @override
  void dispose() {
    _pageController.dispose();
    _videoController?.dispose();
    _timer?.cancel();
    super.dispose();
  }

  void _goToLogin() {
    Navigator.pushReplacement(
      context,
      MaterialPageRoute(builder: (context) => const LoginPage()),
    );
  }

  @override
  Widget build(BuildContext context) {
    final t = AppLocalizations.of(context)!; // للوصول للنصوص المترجمة

    if (loading) {
      return const Scaffold(
        body: Center(child: CircularProgressIndicator()),
      );
    }

    return Scaffold(
      body: Stack(
        fit: StackFit.expand,
        children: [
          // الخلفية (صور أو فيديو)
          onboarding!.type == "video"
              ? _buildVideoBackground()
              : _buildImageBackground(),

          // Overlay أسود خفيف
          Container(color: Colors.black.withOpacity(0.3)),

          // النصوص والأزرار
          _buildTextAndButtons(t),
        ],
      ),
    );
  }

  Widget _buildVideoBackground() {
    return _videoController != null && _videoController!.value.isInitialized
        ? FittedBox(
            fit: BoxFit.cover,
            child: SizedBox(
              width: _videoController!.value.size.width,
              height: _videoController!.value.size.height,
              child: VideoPlayer(_videoController!),
            ),
          )
        : const Center(child: CircularProgressIndicator());
  }

  Widget _buildImageBackground() {
    final images = [
      onboarding!.image1,
      onboarding!.image2,
      onboarding!.image3
    ].where((img) => img != null && img.isNotEmpty).toList();

    return PageView.builder(
      controller: _pageController,
      itemCount: images.length,
      onPageChanged: (index) => setState(() => _currentIndex = index),
      itemBuilder: (context, index) {
        return Image.network(ApiConstants.baseUrl+'/uploads/images/'+images[index], fit: BoxFit.cover);
      },
    );
  }

  Widget _buildTextAndButtons(AppLocalizations t) {
    final locale = Provider.of<LocaleProvider>(context, listen: false).locale;
    final lang = locale.languageCode;

    // اختيار النصوص حسب اللغة
    final caption = lang == 'ar'
        ? (onboarding!.caption_ar ?? "")
        : (onboarding!.caption_en ?? "");

    final description = lang == 'ar'
        ? (onboarding!.description_ar ?? "")
        : (onboarding!.description_en ?? "");

    return Positioned(
      bottom: 60,
      left: 24,
      right: 24,
      child: Column(
        mainAxisAlignment: MainAxisAlignment.end,
        children: [
          Text(
            caption,
            style: const TextStyle(
              color: Colors.white,
              fontSize: 26,
              fontWeight: FontWeight.bold,
            ),
          ),
          const SizedBox(height: 12),
          Text(
            description,
            textAlign: TextAlign.center,
            style: const TextStyle(
              color: Colors.white70,
              fontSize: 16,
            ),
          ),
          const SizedBox(height: 20),
          ElevatedButton(
            onPressed: () {
              Navigator.pushReplacement(
                context,
                MaterialPageRoute(builder: (context) => const SignUpPage()),
              );
            },
            style: ElevatedButton.styleFrom(
              backgroundColor: const Color(0xFF7EC6B1),
              minimumSize: const Size(double.infinity, 48),
              shape: RoundedRectangleBorder(
                borderRadius: BorderRadius.circular(10),
              ),
            ),
            child: Text(
              t.createAccount, // من ملفات الترجمة
              style: const TextStyle(
                  color: Colors.black, fontWeight: FontWeight.bold),
            ),
          ),
          const SizedBox(height: 12),
          GestureDetector(
            onTap: _goToLogin,
            child: Text.rich(
              TextSpan(
                text: t.alreadyHaveAccount + ' ',
                style: const TextStyle(color: Colors.white70),
                children: [
                  TextSpan(
                    text: t.login,
                    style: const TextStyle(
                      color: Color(0xFF7EC6B1),
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                ],
              ),
            ),
          ),
        ],
      ),
    );
  }

}
