import 'package:flutter/material.dart';
import 'package:hockey_app/l10n/app_localizations.dart';
import 'package:hockey_app/services/api_constants.dart';
import '../services/news_service.dart';
import 'content_detail_screen.dart';

class AdsNewsScreen extends StatefulWidget {
  const AdsNewsScreen({super.key});

  @override
  State<AdsNewsScreen> createState() => _AdsNewsScreenState();
}

class _AdsNewsScreenState extends State<AdsNewsScreen> with SingleTickerProviderStateMixin {
  late TabController _tabController;
  late Future<List<Map<String, dynamic>>> _announcementsFuture;
  late Future<List<Map<String, dynamic>>> _newsFuture;

  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 2, vsync: this);

    _announcementsFuture = UpdatesService.getAnnouncements();
    _newsFuture = UpdatesService.getNews();
  }

  @override
  void dispose() {
    _tabController.dispose();
    super.dispose();
  }

  Widget _buildCard(Map<String, dynamic> item) {
    final isArabic = Localizations.localeOf(context).languageCode == "ar";
    final title = isArabic ? (item['title_ar'] ?? '') : (item['title_en'] ?? '');
    final description = isArabic ? (item['description_ar'] ?? '') : (item['description_en'] ?? '');

    return GestureDetector(
      onTap: () {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (_) => AdvDetailScreen(
              title: title,
              description: description,
              image: item['image'],
            ),
          ),
        );
      },
      child: Container(
        margin: const EdgeInsets.symmetric(vertical: 10, horizontal: 16),
        decoration: BoxDecoration(
          color: Theme.of(context).cardColor,
          borderRadius: BorderRadius.circular(12),
          boxShadow: [
            BoxShadow(
              color: Colors.black.withOpacity(0.2),
              blurRadius: 6,
              offset: const Offset(0, 4),
            ),
          ],
        ),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Image.network(
              ApiConstants.baseUrl + '/uploads/images/' + item['image'],
              width: double.infinity,
              height: 160,
              fit: BoxFit.cover,
            ),
            Padding(
              padding: const EdgeInsets.all(12.0),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    title,
                    style: Theme.of(context)
                        .textTheme
                        .titleLarge
                        ?.copyWith(fontWeight: FontWeight.bold),
                  ),
                  const SizedBox(height: 6),
                  Text(
                    description,
                    maxLines: 2,
                    overflow: TextOverflow.ellipsis,
                    style: Theme.of(context)
                        .textTheme
                        .labelLarge
                        ?.copyWith(color: Colors.grey[600]),
                  ),
                ],
              ),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildFuture(Future<List<Map<String, dynamic>>> future) {
    final t = AppLocalizations.of(context)!;
    return FutureBuilder<List<Map<String, dynamic>>>(
      future: future,
      builder: (context, snapshot) {
        if (snapshot.connectionState == ConnectionState.waiting) {
          return const Center(child: CircularProgressIndicator());
        } else if (snapshot.hasError) {
          return Center(child: Text("${t.error}: ${snapshot.error}"));
        } else if (!snapshot.hasData || snapshot.data!.isEmpty) {
          return Center(child: Text(t.noDataAvailable));
        }
        return ListView(
          children: snapshot.data!.map((item) => _buildCard(item)).toList(),
        );
      },
    );
  }

  @override
  Widget build(BuildContext context) {
    final t = AppLocalizations.of(context)!;
    return Scaffold(
      appBar: AppBar(
        title: Text(t.updates),
        bottom: TabBar(
          controller: _tabController,
          tabs: [
            Tab(icon: const Icon(Icons.campaign), text: t.announcements),
            Tab(icon: const Icon(Icons.feed), text: t.news),
          ],
        ),
      ),
      body: TabBarView(
        controller: _tabController,
        children: [
          _buildFuture(_announcementsFuture),
          _buildFuture(_newsFuture),
        ],
      ),
    );
  }
}
