import 'package:flutter/material.dart';
import 'package:youtube_player_flutter/youtube_player_flutter.dart';
import '../l10n/app_localizations.dart';

class MatchDetailScreen extends StatefulWidget {
  final Map<String, dynamic> match;
  const MatchDetailScreen({super.key, required this.match});

  @override
  State<MatchDetailScreen> createState() => _MatchDetailScreenState();
}

class _MatchDetailScreenState extends State<MatchDetailScreen> {
  late YoutubePlayerController _youtubeController;
  bool _isYoutubeLink = false;
  String? _error;

  @override
  void initState() {
    super.initState();
    _initializePlayer();
  }

  void _initializePlayer() {
    final videoUrl = widget.match['video'] ?? "";

    if (videoUrl.isEmpty) {
      setState(() {
        _error = "لا يوجد فيديو لهذه المباراة";
        _isYoutubeLink = false;
      });
      return;
    }

    String? videoId;
    if (videoUrl.contains("watch?v=")) {
      videoId = YoutubePlayer.convertUrlToId(videoUrl);
    } else if (videoUrl.contains("/live/")) {
      videoId = videoUrl.split("/live/").last;
    }

    if (videoId == null || videoId.isEmpty) {
      setState(() {
        _error = "الرابط غير صالح أو غير مدعوم";
        _isYoutubeLink = false;
      });
      return;
    }

    _isYoutubeLink = true;
    _youtubeController = YoutubePlayerController(
      initialVideoId: videoId,
      flags: const YoutubePlayerFlags(
        autoPlay: true,
        mute: false,
        enableCaption: true,
        isLive: true,
      ),
    );
    _error = null;
  }

  @override
  void dispose() {
    if (_isYoutubeLink) {
      _youtubeController.dispose();
    }
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final t = AppLocalizations.of(context)!;
    final isArabic = Localizations.localeOf(context).languageCode == "ar";

    final teamA = isArabic
        ? (widget.match['teamA_ar'] ?? '')
        : (widget.match['teamA_en'] ?? '');
    final teamB = isArabic
        ? (widget.match['teamB_ar'] ?? '')
        : (widget.match['teamB_en'] ?? '');
    final description = isArabic
        ? (widget.match['description_ar'] ?? '')
        : (widget.match['description_en'] ?? '');

    return Scaffold(
      backgroundColor: const Color(0xFF1E1F22),
      appBar: AppBar(
        title: Text("$teamA vs $teamB"),
        backgroundColor: const Color(0xFF1E1F22),
        centerTitle: true,
      ),
      body: SingleChildScrollView(
        child: Column(
          children: [
            // فيديو المباراة
            Container(
              height: 250,
              width: double.infinity,
              color: Colors.black,
              child: _error != null
                  ? Center(
                child: Column(
                  mainAxisAlignment: MainAxisAlignment.center,
                  children: [
                    Text(
                      _error!,
                      style: const TextStyle(color: Colors.white),
                      textAlign: TextAlign.center,
                    ),
                    const SizedBox(height: 12),
                    ElevatedButton(
                      onPressed: _initializePlayer,
                      style: ElevatedButton.styleFrom(
                        backgroundColor: Colors.red,
                      ),
                      child: const Text("إعادة المحاولة"),
                    ),
                  ],
                ),
              )
                  : _isYoutubeLink
                  ? YoutubePlayer(
                controller: _youtubeController,
                showVideoProgressIndicator: true,
                progressIndicatorColor: Colors.red,
              )
                  : const Center(
                child:
                CircularProgressIndicator(color: Colors.white),
              ),
            ),
            const SizedBox(height: 16),

            // تفاصيل المباراة
            Padding(
              padding: const EdgeInsets.symmetric(horizontal: 16),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  // الفرق والنتيجة
                  if (teamA.isNotEmpty ||
                      teamB.isNotEmpty ||
                      (widget.match['score']?.toString().isNotEmpty ?? false))
                    Card(
                      color: const Color(0xFF2C2D33),
                      shape: RoundedRectangleBorder(
                        borderRadius: BorderRadius.circular(16),
                      ),
                      elevation: 4,
                      child: Padding(
                        padding: const EdgeInsets.all(16),
                        child: Row(
                          mainAxisAlignment: MainAxisAlignment.spaceBetween,
                          children: [
                            Column(
                              crossAxisAlignment: CrossAxisAlignment.start,
                              children: [
                                Text(teamA,
                                    style: const TextStyle(
                                        color: Colors.white,
                                        fontSize: 18,
                                        fontWeight: FontWeight.bold)),
                                const SizedBox(height: 4),
                                Text(t.score,
                                    style:
                                    const TextStyle(color: Colors.grey)),
                              ],
                            ),
                            Text(
                              widget.match['score'] ?? '-',
                              style: const TextStyle(
                                  color: Colors.yellow,
                                  fontSize: 20,
                                  fontWeight: FontWeight.bold),
                            ),
                            Column(
                              crossAxisAlignment: CrossAxisAlignment.end,
                              children: [
                                Text(teamB,
                                    style: const TextStyle(
                                        color: Colors.white,
                                        fontSize: 18,
                                        fontWeight: FontWeight.bold)),
                                const SizedBox(height: 4),
                                Text('',
                                    style: const TextStyle(color: Colors.grey)),
                              ],
                            ),
                          ],
                        ),
                      ),
                    ),

                  const SizedBox(height: 16),

                  // وصف المباراة
                  if (description.isNotEmpty)
                    Card(
                      color: const Color(0xFF2C2D33),
                      shape: RoundedRectangleBorder(
                        borderRadius: BorderRadius.circular(16),
                      ),
                      elevation: 4,
                      child: Padding(
                        padding: const EdgeInsets.all(16),
                        child: Column(
                          crossAxisAlignment: CrossAxisAlignment.start,
                          children: [
                            Text(t.details,
                                style: const TextStyle(
                                    color: Colors.white,
                                    fontSize: 16,
                                    fontWeight: FontWeight.bold)),
                            const SizedBox(height: 8),
                            Text(description,
                                style: const TextStyle(color: Colors.grey)),
                          ],
                        ),
                      ),
                    ),
                ],
              ),
            ),

            const SizedBox(height: 24),
          ],
        ),
      ),
    );
  }
}
