import 'dart:async';
import 'package:flutter/material.dart';
import 'package:youtube_player_flutter/youtube_player_flutter.dart';
import 'package:intl/intl.dart';
import 'package:hockey_app/l10n/app_localizations.dart';
import 'package:hockey_app/screens/match_detail.dart';
import 'package:hockey_app/services/api_constants.dart';
import '../services/live_stream_service.dart';

class LiveStreamScreen extends StatefulWidget {
  const LiveStreamScreen({super.key});

  @override
  State<LiveStreamScreen> createState() => _LiveStreamScreenState();
}

class _LiveStreamScreenState extends State<LiveStreamScreen> {
  YoutubePlayerController? _youtubeController;
  bool _isLoading = true;
  String? _error;

  String? nowLiveTitle;
  String? nowLiveUrl;
  List<Map<String, String>> upcoming = [];
  List<Map<String, String>> previousLive = [];

  Timer? _timer;
  Duration _remaining = Duration.zero;
  DateTime? _nextStreamTime;

  @override
  void initState() {
    super.initState();
    _loadStreams();
  }

  String _localizedValue(Map<String, String> item, String baseKey, String lang) {
    return item["${baseKey}_$lang"] ?? item["${baseKey}_en"] ?? "";
  }

  String _formatDateTime(String raw) {
    try {
      final dt = DateTime.tryParse(raw);
      if (dt == null) return raw;
      final localDt = dt.toLocal();
      return DateFormat('EEEE dd MMMM yyyy - hh:mm a', 'ar').format(localDt);
    } catch (_) {
      return raw;
    }
  }

  String? _extractYoutubeId(String url) {
    final regex = RegExp(r'(?:v=|\/)([0-9A-Za-z_-]{11}).*', caseSensitive: false);
    final match = regex.firstMatch(url);
    return match != null ? match.group(1) : null;
  }

  Future<void> _loadStreams() async {
    try {
      final data = await LiveStreamService().getStreams();

      setState(() {
        nowLiveTitle = data["nowLive"]?["title"];
        nowLiveUrl = data["nowLive"]?["streamUrl"];
        upcoming = (data["upcoming"] as List<dynamic>? ?? [])
            .map((e) => (e as Map<String, dynamic>).map((k, v) => MapEntry(k, v.toString())))
            .toList();
        previousLive = (data["previousLive"] as List<dynamic>? ?? [])
            .map((e) => (e as Map<String, dynamic>).map((k, v) => MapEntry(k, v.toString())))
            .toList();
      });

      if (nowLiveUrl != null && nowLiveUrl!.isNotEmpty) {
        final videoId = _extractYoutubeId(nowLiveUrl!);
        if (videoId != null) {
          _youtubeController = YoutubePlayerController(
            initialVideoId: videoId,
            flags: const YoutubePlayerFlags(autoPlay: true, isLive: true, mute: false),
          );
        } else {
          setState(() {
            _error = "الرابط غير صالح";
          });
        }
        setState(() => _isLoading = false);
      } else if (upcoming.isNotEmpty) {
        try {
          _nextStreamTime = DateTime.parse(upcoming.first["time"] ?? "");
        } catch (_) {
          _nextStreamTime = null;
        }
        if (_nextStreamTime != null) _startCountdown();
        setState(() => _isLoading = false);
      } else {
        setState(() => _isLoading = false);
      }
    } catch (e) {
      setState(() {
        _isLoading = false;
        _error = "فشل تحميل البث: $e";
      });
    }
  }

  void _startCountdown() {
    _timer?.cancel();
    if (_nextStreamTime == null) return;

    _timer = Timer.periodic(const Duration(seconds: 1), (_) {
      final now = DateTime.now();
      if (now.isBefore(_nextStreamTime!)) {
        setState(() => _remaining = _nextStreamTime!.difference(now));
      } else {
        _timer?.cancel();
        _remaining = Duration.zero;
        _loadStreams();
      }
    });
  }

  String _formatDuration(Duration duration) {
    String twoDigits(int n) => n.toString().padLeft(2, "0");
    return "${twoDigits(duration.inHours)}:${twoDigits(duration.inMinutes % 60)}:${twoDigits(duration.inSeconds % 60)}";
  }

  @override
  void dispose() {
    _youtubeController?.dispose();
    _timer?.cancel();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final tr = AppLocalizations.of(context)!;
    final lang = Localizations.localeOf(context).languageCode;

    return Scaffold(
      backgroundColor: Theme.of(context).scaffoldBackgroundColor,
      appBar: AppBar(
        title: Text(tr.live_stream),
        centerTitle: true,
        elevation: 0,
        flexibleSpace: Container(
          decoration: const BoxDecoration(
            gradient: LinearGradient(
              begin: Alignment.topLeft,
              end: Alignment.bottomRight,
              stops: [0.0, 0.9],
              colors: [Color(0xFF4E277B), Color(0xFF7EC6B1)],
            ),
          ),
        ),
        backgroundColor: Colors.white,
        foregroundColor: Colors.white,
      ),
      body: SingleChildScrollView(
        padding: const EdgeInsets.fromLTRB(16, 8, 16, 16),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Text(tr.now_live, style: Theme.of(context).textTheme.titleLarge),
            const SizedBox(height: 10),
            Container(
              height: 200,
              decoration: BoxDecoration(borderRadius: BorderRadius.circular(16), color: Colors.black),
              child: _error != null
                  ? Center(child: Text(_error!, style: const TextStyle(color: Colors.white)))
                  : _isLoading
                  ? const Center(child: CircularProgressIndicator())
                  : _youtubeController != null
                  ? ClipRRect(
                borderRadius: BorderRadius.circular(16),
                child: YoutubePlayer(controller: _youtubeController!),
              )
                  : _buildNoLivePlaceholder(tr),
            ),
            const SizedBox(height: 20),
            if (upcoming.isNotEmpty)
              Text(tr.upcoming_streams, style: Theme.of(context).textTheme.titleLarge),
            const SizedBox(height: 10),
            Column(
              children: List.generate(upcoming.length, (index) {
                final item = upcoming[index];
                return _buildUpcomingCard(
                  context,
                  index,
                  _localizedValue(item, "title", lang),
                  item["time"] ?? "",
                  item["thumbnail"] ?? "",
                  tr,
                );
              }),
            ),
            const SizedBox(height: 20),
            if (previousLive.isNotEmpty)
              Text(tr.previous_streams, style: Theme.of(context).textTheme.titleLarge),
            const SizedBox(height: 10),
            Column(
              children: List.generate(previousLive.length, (index) {
                final item = previousLive[index];
                return _buildPreviousCard(
                  context,
                  index,
                  _localizedValue(item, "title", lang),
                  item["time"] ?? item["startTime"] ?? "",
                  item["thumbnail"] ?? "",
                  item["streamUrl"] ?? "",
                  tr,
                  _localizedValue(item, "teamA", lang),
                  _localizedValue(item, "teamB", lang),
                  item["score"] ?? "",
                  _localizedValue(item, "description", lang),
                );
              }),
            ),
          ],
        ),
      ),
    );
  }

  Widget _buildNoLivePlaceholder(AppLocalizations tr) {
    return Stack(
      fit: StackFit.expand,
      children: [
        Image.asset("assets/images/logo_landingpage_NoBackground.png", fit: BoxFit.cover),
        Container(
          decoration: BoxDecoration(color: Colors.black.withOpacity(0.4), borderRadius: BorderRadius.circular(16)),
        ),
        Center(
          child: Column(
            mainAxisAlignment: MainAxisAlignment.center,
            children: [
              Text(tr.no_live_stream, style: const TextStyle(color: Colors.white, fontSize: 18)),
              const SizedBox(height: 8),
              if (_nextStreamTime != null && _remaining > Duration.zero)
                Column(
                  children: [
                    Text(tr.next_stream_starts_in, style: const TextStyle(color: Colors.white, fontSize: 16)),
                    const SizedBox(height: 5),
                    Text(
                      _formatDuration(_remaining),
                      style: const TextStyle(color: Colors.yellow, fontSize: 24, fontWeight: FontWeight.bold),
                    ),
                  ],
                ),
            ],
          ),
        ),
      ],
    );
  }

  Widget _buildUpcomingCard(BuildContext context, int index, String title, String time, String imagePath, AppLocalizations tr) {
    return Container(
      margin: const EdgeInsets.symmetric(vertical: 8),
      decoration: BoxDecoration(color: Colors.white, borderRadius: BorderRadius.circular(12), boxShadow: [
        BoxShadow(color: Colors.white.withOpacity(0.6), blurRadius: 8, offset: const Offset(0, 4)),
      ]),
      child: ListTile(
        contentPadding: const EdgeInsets.all(12),
        leading: ClipRRect(
          borderRadius: BorderRadius.circular(8),
          child: Image.network(
            ApiConstants.baseUrl + '/uploads/images/' + imagePath,
            width: 60,
            height: 60,
            fit: BoxFit.cover,
            errorBuilder: (_, __, ___) => const Icon(Icons.broken_image, size: 40),
          ),
        ),
        title: Text(title, style: const TextStyle(color: Colors.black, fontWeight: FontWeight.bold)),
        subtitle: Text(_formatDateTime(time), style: const TextStyle(color: Colors.black)),
        trailing: ElevatedButton(
          style: ElevatedButton.styleFrom(backgroundColor: const Color(0xFF7EC6B1)),
          onPressed: () {
            ScaffoldMessenger.of(context).showSnackBar(SnackBar(content: Text(tr.stream_not_started)));
          },
          child: Text(tr.watch_now),
        ),
      ),
    );
  }

  Widget _buildPreviousCard(
      BuildContext context,
      int index,
      String title,
      String time,
      String imagePath,
      String streamUrl,
      AppLocalizations tr,
      String teamA,
      String teamB,
      String score,
      String description,
      ) {
    return Container(
      margin: const EdgeInsets.symmetric(vertical: 8),
      decoration: BoxDecoration(color: Colors.white, borderRadius: BorderRadius.circular(12), boxShadow: [
        BoxShadow(color: Colors.white.withOpacity(0.6), blurRadius: 8, offset: const Offset(0, 4)),
      ]),
      child: ListTile(
        contentPadding: const EdgeInsets.all(12),
        leading: ClipRRect(
          borderRadius: BorderRadius.circular(8),
          child: Image.network(
            ApiConstants.baseUrl + '/uploads/images/' + imagePath,
            width: 60,
            height: 60,
            fit: BoxFit.cover,
            errorBuilder: (_, __, ___) => const Icon(Icons.broken_image, size: 40),
          ),
        ),
        title: Text(title, style: const TextStyle(color: Colors.black, fontWeight: FontWeight.bold)),
        subtitle: Text(_formatDateTime(time), style: const TextStyle(color: Colors.black)),
        trailing: ElevatedButton(
          style: ElevatedButton.styleFrom(backgroundColor: const Color(0xFF7EC6B1)),
          onPressed: () {
            Navigator.push(
              context,
              MaterialPageRoute(
                builder: (_) => MatchDetailScreen(
                  match: {
                    "teamA": teamA,
                    "teamB": teamB,
                    "score": score,
                    "date": time,
                    "thumbnail": imagePath,
                    "video": streamUrl,
                    "description": description,
                  },
                ),  
              ),
            );
          },
          child: Text(tr.watch_now),
        ),
      ),
    );
  }
}





// import 'dart:async';
// import 'package:flutter/material.dart';
// import 'package:youtube_player_flutter/youtube_player_flutter.dart';
// import 'package:intl/intl.dart';
// import 'package:hockey_app/l10n/app_localizations.dart';
// import 'package:hockey_app/screens/match_detail.dart';
// import 'package:hockey_app/services/api_constants.dart';
// import '../services/live_stream_service.dart';
//
// class LiveStreamScreen extends StatefulWidget {
//   const LiveStreamScreen({super.key});
//
//   @override
//   State<LiveStreamScreen> createState() => _LiveStreamScreenState();
// }
//
// class _LiveStreamScreenState extends State<LiveStreamScreen> with TickerProviderStateMixin {
//   YoutubePlayerController? _youtubeController;
//   bool _isLoading = true;
//   String? _error;
//
//   String? nowLiveTitle;
//   String? nowLiveUrl;
//   List<Map<String, String>> upcoming = [];
//   List<Map<String, String>> previousLive = [];
//
//   Timer? _timer;
//   Duration _remaining = Duration.zero;
//   DateTime? _nextStreamTime;
//
//   late AnimationController _mainController;
//   late Animation<double> _videoFadeAnimation;
//   late Animation<Offset> _videoSlideAnimation;
//   late Animation<double> _nowLiveFadeAnimation;
//   late Animation<Offset> _nowLiveSlideAnimation;
//   late Animation<double> _sectionFadeAnimation;
//   late Animation<Offset> _sectionSlideAnimation;
//
//   late List<AnimationController> _upcomingControllers;
//   late List<Animation<double>> _upcomingFadeAnimations;
//   late List<Animation<Offset>> _upcomingSlideAnimations;
//
//   late List<AnimationController> _previousControllers;
//   late List<Animation<double>> _previousFadeAnimations;
//   late List<Animation<Offset>> _previousSlideAnimations;
//
//   @override
//   void initState() {
//     super.initState();
//     _setupMainAnimations();
//     _loadStreams();
//   }
//
//   /// Helper: pick localized value
//   String _localizedValue(Map<String, String> item, String baseKey, String lang) {
//     return item["${baseKey}_$lang"] ?? item["${baseKey}_en"] ?? "";
//   }
//
//   String _formatDateTime(String raw) {
//     try {
//       final dt = DateTime.tryParse(raw);
//       if (dt == null) return raw;
//       final localDt = dt.toLocal();
//       return DateFormat('EEEE dd MMMM yyyy - hh:mm a', 'ar').format(localDt);
//     } catch (_) {
//       return raw;
//     }
//   }
//
//   void _setupMainAnimations() {
//     _mainController = AnimationController(duration: const Duration(milliseconds: 1800), vsync: this);
//     _videoFadeAnimation = Tween<double>(begin: 0, end: 1)
//         .animate(CurvedAnimation(parent: _mainController, curve: Curves.easeIn));
//     _videoSlideAnimation = Tween<Offset>(begin: const Offset(0, 0.6), end: Offset.zero)
//         .animate(CurvedAnimation(parent: _mainController, curve: Curves.easeOut));
//     _nowLiveFadeAnimation = Tween<double>(begin: 0, end: 1).animate(
//         CurvedAnimation(parent: _mainController, curve: const Interval(0.3, 1.0, curve: Curves.easeIn)));
//     _nowLiveSlideAnimation = Tween<Offset>(begin: const Offset(0, 0.6), end: Offset.zero)
//         .animate(CurvedAnimation(parent: _mainController, curve: const Interval(0.3, 1.0, curve: Curves.easeOut)));
//     _sectionFadeAnimation = Tween<double>(begin: 0, end: 1)
//         .animate(CurvedAnimation(parent: _mainController, curve: Curves.easeOut));
//     _sectionSlideAnimation = Tween<Offset>(begin: const Offset(0, 0.5), end: Offset.zero)
//         .animate(CurvedAnimation(parent: _mainController, curve: Curves.easeOut));
//     _mainController.forward();
//   }
//
//   void _setupCardAnimations() {
//     _upcomingControllers = List.generate(
//       upcoming.length,
//           (index) => AnimationController(duration: const Duration(milliseconds: 800), vsync: this),
//     );
//     _upcomingFadeAnimations = _upcomingControllers
//         .map((c) => Tween<double>(begin: 0, end: 1).animate(CurvedAnimation(parent: c, curve: Curves.easeOut)))
//         .toList();
//     _upcomingSlideAnimations = _upcomingControllers
//         .map((c) => Tween<Offset>(begin: const Offset(0, 0.5), end: Offset.zero)
//         .animate(CurvedAnimation(parent: c, curve: Curves.easeOut)))
//         .toList();
//     for (int i = 0; i < _upcomingControllers.length; i++) {
//       Future.delayed(Duration(milliseconds: 300 * i), () {
//         if (mounted) _upcomingControllers[i].forward();
//       });
//     }
//
//     _previousControllers = List.generate(
//       previousLive.length,
//           (index) => AnimationController(duration: const Duration(milliseconds: 800), vsync: this),
//     );
//     _previousFadeAnimations = _previousControllers
//         .map((c) => Tween<double>(begin: 0, end: 1).animate(CurvedAnimation(parent: c, curve: Curves.easeOut)))
//         .toList();
//     _previousSlideAnimations = _previousControllers
//         .map((c) => Tween<Offset>(begin: const Offset(0, 0.5), end: Offset.zero)
//         .animate(CurvedAnimation(parent: c, curve: Curves.easeOut)))
//         .toList();
//     for (int i = 0; i < _previousControllers.length; i++) {
//       Future.delayed(Duration(milliseconds: 300 * i), () {
//         if (mounted) _previousControllers[i].forward();
//       });
//     }
//   }
//
//   String? _extractYoutubeId(String url) {
//     final regex = RegExp(r'(?:v=|\/)([0-9A-Za-z_-]{11}).*', caseSensitive: false);
//     final match = regex.firstMatch(url);
//     return match != null ? match.group(1) : null;
//   }
//
//   Future<void> _loadStreams() async {
//     try {
//       final data = await LiveStreamService().getStreams();
//
//       setState(() {
//         nowLiveTitle = data["nowLive"]?["title"];
//         nowLiveUrl = data["nowLive"]?["streamUrl"];
//         upcoming = (data["upcoming"] as List<dynamic>? ?? [])
//             .map((e) => (e as Map<String, dynamic>).map((k, v) => MapEntry(k, v.toString())))
//             .toList();
//         previousLive = (data["previousLive"] as List<dynamic>? ?? [])
//             .map((e) => (e as Map<String, dynamic>).map((k, v) => MapEntry(k, v.toString())))
//             .toList();
//       });
//
//       _setupCardAnimations();
//
//       if (nowLiveUrl != null && nowLiveUrl!.isNotEmpty) {
//         final videoId = _extractYoutubeId(nowLiveUrl!);
//         if (videoId != null) {
//           _youtubeController = YoutubePlayerController(
//             initialVideoId: videoId,
//             flags: const YoutubePlayerFlags(autoPlay: true, isLive: true, mute: false),
//           );
//         } else {
//           setState(() {
//             _error = "الرابط غير صالح";
//           });
//         }
//         setState(() => _isLoading = false);
//       } else if (upcoming.isNotEmpty) {
//         try {
//           _nextStreamTime = DateTime.parse(upcoming.first["time"] ?? "");
//         } catch (_) {
//           _nextStreamTime = null;
//         }
//         if (_nextStreamTime != null) _startCountdown();
//         setState(() => _isLoading = false);
//       } else {
//         setState(() => _isLoading = false);
//       }
//     } catch (e) {
//       setState(() {
//         _isLoading = false;
//         _error = "فشل تحميل البث: $e";
//       });
//     }
//   }
//
//   void _startCountdown() {
//     _timer?.cancel();
//     if (_nextStreamTime == null) return;
//
//     _timer = Timer.periodic(const Duration(seconds: 1), (_) {
//       final now = DateTime.now();
//       if (now.isBefore(_nextStreamTime!)) {
//         setState(() => _remaining = _nextStreamTime!.difference(now));
//       } else {
//         _timer?.cancel();
//         _remaining = Duration.zero;
//         _loadStreams();
//       }
//     });
//   }
//
//   String _formatDuration(Duration duration) {
//     String twoDigits(int n) => n.toString().padLeft(2, "0");
//     return "${twoDigits(duration.inHours)}:${twoDigits(duration.inMinutes % 60)}:${twoDigits(duration.inSeconds % 60)}";
//   }
//
//   @override
//   void dispose() {
//     _mainController.dispose();
//     _youtubeController?.dispose();
//     _timer?.cancel();
//     for (final c in _upcomingControllers) c.dispose();
//     for (final c in _previousControllers) c.dispose();
//     super.dispose();
//   }
//
//   @override
//   Widget build(BuildContext context) {
//     final tr = AppLocalizations.of(context)!;
//     final lang = Localizations.localeOf(context).languageCode;
//
//     return Scaffold(
//       backgroundColor: Theme.of(context).scaffoldBackgroundColor,
//       appBar: AppBar(
//         title: Text(tr.live_stream),
//         centerTitle: true,
//         elevation: 0,
//         flexibleSpace: Container(
//           decoration: const BoxDecoration(
//             gradient: LinearGradient(
//               begin: Alignment.topLeft,
//               end: Alignment.bottomRight,
//               stops: [0.0, 0.9],
//               colors: [Color(0xFF4E277B), Color(0xFF7EC6B1)],
//             ),
//           ),
//         ),
//         backgroundColor: Colors.white,
//         foregroundColor: Colors.white,
//       ),
//       body: SingleChildScrollView(
//         padding: const EdgeInsets.fromLTRB(16, 8, 16, 16),
//         child: Column(
//           crossAxisAlignment: CrossAxisAlignment.start,
//           children: [
//             FadeTransition(
//               opacity: _nowLiveFadeAnimation,
//               child: SlideTransition(
//                 position: _nowLiveSlideAnimation,
//                 child: Text(tr.now_live, style: Theme.of(context).textTheme.titleLarge),
//               ),
//             ),
//             const SizedBox(height: 10),
//             FadeTransition(
//               opacity: _videoFadeAnimation,
//               child: SlideTransition(
//                 position: _videoSlideAnimation,
//                 child: Container(
//                   height: 200,
//                   decoration: BoxDecoration(borderRadius: BorderRadius.circular(16), color: Colors.black),
//                   child: _error != null
//                       ? Center(child: Text(_error!, style: const TextStyle(color: Colors.white)))
//                       : _isLoading
//                       ? const Center(child: CircularProgressIndicator())
//                       : _youtubeController != null
//                       ? ClipRRect(
//                     borderRadius: BorderRadius.circular(16),
//                     child: YoutubePlayer(controller: _youtubeController!),
//                   )
//                       : _buildNoLivePlaceholder(tr),
//                 ),
//               ),
//             ),
//             const SizedBox(height: 20),
//             if (upcoming.isNotEmpty)
//               FadeTransition(
//                 opacity: _sectionFadeAnimation,
//                 child: SlideTransition(
//                   position: _sectionSlideAnimation,
//                   child: Text(tr.upcoming_streams, style: Theme.of(context).textTheme.titleLarge),
//                 ),
//               ),
//             const SizedBox(height: 10),
//             Column(
//               children: List.generate(upcoming.length, (index) {
//                 final item = upcoming[index];
//                 return FadeTransition(
//                   opacity: _upcomingFadeAnimations[index],
//                   child: SlideTransition(
//                     position: _upcomingSlideAnimations[index],
//                     child: _buildUpcomingCard(
//                       context,
//                       index,
//                       _localizedValue(item, "title", lang),
//                       item["time"] ?? "",
//                       item["thumbnail"] ?? "",
//                       tr,
//                     ),
//                   ),
//                 );
//               }),
//             ),
//             const SizedBox(height: 20),
//             if (previousLive.isNotEmpty)
//               FadeTransition(
//                 opacity: _sectionFadeAnimation,
//                 child: SlideTransition(
//                   position: _sectionSlideAnimation,
//                   child: Text(tr.previous_streams, style: Theme.of(context).textTheme.titleLarge),
//                 ),
//               ),
//             const SizedBox(height: 10),
//             Column(
//               children: List.generate(previousLive.length, (index) {
//                 final item = previousLive[index];
//                 return FadeTransition(
//                   opacity: _previousFadeAnimations[index],
//                   child: SlideTransition(
//                     position: _previousSlideAnimations[index],
//                     child: _buildPreviousCard(
//                       context,
//                       index,
//                       _localizedValue(item, "title", lang),
//                       item["time"] ?? item["startTime"] ?? "",
//                       item["thumbnail"] ?? "",
//                       item["streamUrl"] ?? "",
//                       tr,
//                       _localizedValue(item, "teamA", lang),
//                       _localizedValue(item, "teamB", lang),
//                       item["score"] ?? "",
//                       _localizedValue(item, "description", lang),
//                     ),
//                   ),
//                 );
//               }),
//             ),
//           ],
//         ),
//       ),
//     );
//   }
//
//   Widget _buildNoLivePlaceholder(AppLocalizations tr) {
//     return Stack(
//       fit: StackFit.expand,
//       children: [
//         Image.asset("assets/images/logo_landingpage_NoBackground.png", fit: BoxFit.cover),
//         Container(
//           decoration: BoxDecoration(color: Colors.black.withOpacity(0.4), borderRadius: BorderRadius.circular(16)),
//         ),
//         Center(
//           child: Column(
//             mainAxisAlignment: MainAxisAlignment.center,
//             children: [
//               Text(tr.no_live_stream, style: const TextStyle(color: Colors.white, fontSize: 18)),
//               const SizedBox(height: 8),
//               if (_nextStreamTime != null && _remaining > Duration.zero)
//                 Column(
//                   children: [
//                     Text(tr.next_stream_starts_in, style: const TextStyle(color: Colors.white, fontSize: 16)),
//                     const SizedBox(height: 5),
//                     Text(
//                       _formatDuration(_remaining),
//                       style: const TextStyle(color: Colors.yellow, fontSize: 24, fontWeight: FontWeight.bold),
//                     ),
//                   ],
//                 ),
//             ],
//           ),
//         ),
//       ],
//     );
//   }
//
//   Widget _buildUpcomingCard(BuildContext context, int index, String title, String time, String imagePath, AppLocalizations tr) {
//     return Container(
//       margin: const EdgeInsets.symmetric(vertical: 8),
//       decoration: BoxDecoration(color: Colors.white, borderRadius: BorderRadius.circular(12), boxShadow: [
//         BoxShadow(color: Colors.white.withOpacity(0.6), blurRadius: 8, offset: const Offset(0, 4)),
//       ]),
//       child: ListTile(
//         contentPadding: const EdgeInsets.all(12),
//         leading: ClipRRect(
//           borderRadius: BorderRadius.circular(8),
//           child: Image.network(
//             ApiConstants.baseUrl + '/uploads/images/' + imagePath,
//             width: 60,
//             height: 60,
//             fit: BoxFit.cover,
//             errorBuilder: (_, __, ___) => const Icon(Icons.broken_image, size: 40),
//           ),
//         ),
//         title: Text(title, style: const TextStyle(color: Colors.black, fontWeight: FontWeight.bold)),
//         subtitle: Text(_formatDateTime(time), style: const TextStyle(color: Colors.black)),
//         trailing: ElevatedButton(
//           style: ElevatedButton.styleFrom(backgroundColor: const Color(0xFF7EC6B1)),
//           onPressed: () {
//             ScaffoldMessenger.of(context).showSnackBar(SnackBar(content: Text(tr.stream_not_started)));
//           },
//           child: Text(tr.watch_now),
//         ),
//       ),
//     );
//   }
//
//   Widget _buildPreviousCard(
//       BuildContext context,
//       int index,
//       String title,
//       String time,
//       String imagePath,
//       String streamUrl,
//       AppLocalizations tr,
//       String teamA,
//       String teamB,
//       String score,
//       String description,
//       ) {
//     return Container(
//       margin: const EdgeInsets.symmetric(vertical: 8),
//       decoration: BoxDecoration(color: Colors.white, borderRadius: BorderRadius.circular(12), boxShadow: [
//         BoxShadow(color: Colors.white.withOpacity(0.6), blurRadius: 8, offset: const Offset(0, 4)),
//       ]),
//       child: ListTile(
//         contentPadding: const EdgeInsets.all(12),
//         leading: ClipRRect(
//           borderRadius: BorderRadius.circular(8),
//           child: Image.network(
//             ApiConstants.baseUrl + '/uploads/images/' + imagePath,
//             width: 60,
//             height: 60,
//             fit: BoxFit.cover,
//             errorBuilder: (_, __, ___) => const Icon(Icons.broken_image, size: 40),
//           ),
//         ),
//         title: Text(title, style: const TextStyle(color: Colors.black, fontWeight: FontWeight.bold)),
//         subtitle: Text(_formatDateTime(time), style: const TextStyle(color: Colors.black)),
//         trailing: ElevatedButton(
//           style: ElevatedButton.styleFrom(backgroundColor: const Color(0xFF7EC6B1)),
//           onPressed: () {
//             Navigator.push(
//               context,
//               MaterialPageRoute(
//                 builder: (_) => MatchDetailScreen(
//                   match: {
//                     "teamA": teamA,
//                     "teamB": teamB,
//                     "score": score,
//                     "date": time,
//                     "thumbnail": imagePath,
//                     "video": streamUrl,
//                     "description": description,
//                   },
//                 ),
//               ),
//             );
//           },
//           child: Text(tr.watch_now),
//         ),
//       ),
//     );
//   }
// }
