import 'package:flutter/material.dart';
import 'package:hockey_app/services/api_constants.dart';
import 'package:hockey_app/services/user_service.dart';
import '../models/exercise_model.dart';
import '../models/user_model.dart';
import '../services/exercise_service.dart';
import '../services/home_page_service.dart';
import 'AdsHorizontalList.dart';
import 'bio.dart';
import 'exercise_detail.dart';
import 'matches.dart';
import 'settings_screen.dart';
import 'about_us_page.dart';
import 'training_details_page.dart';
import '../main.dart';
import 'package:carousel_slider/carousel_slider.dart';
import '../l10n/app_localizations.dart';

class HomeScreen extends StatefulWidget {
  final String username;
  const HomeScreen({super.key, this.username = "User"});

  @override
  State<HomeScreen> createState() => _HomeScreenState();
}

class _HomeScreenState extends State<HomeScreen>
    with TickerProviderStateMixin, RouteAware {
  late AnimationController _controller;
  late Animation<double> _scale1;
  late Animation<double> _scale2;
  late Animation<double> _scale3;

  late Animation<double> _fade1;
  late Animation<double> _fade2;
  late Animation<double> _fade3;

  late Animation<Offset> _slide1;
  late Animation<Offset> _slide2;
  late Animation<Offset> _slide3;

  Map<int, String> homeImages = {};
  bool isLoading = true;
  List<Exercise> exercises = [];
  User? user;

  Future<void> _loadExercises() async {
    try {
      final List<Exercise> data = await ExerciseService.getExercises();
      setState(() {
        exercises = data;
      });
    } catch (e) {
      debugPrint("Error loading exercises: $e");
    }
  }

  Future<void> _loadUser() async {
    user = await UserService.getUser();
    setState(() {});
  }

  String getInitials(User? user) {
    if (user == null) return "";

    final first = (user.first_name != null && user.first_name!.isNotEmpty)
        ? user.first_name![0].toUpperCase()
        : "";
    final last = (user.last_name != null && user.last_name!.isNotEmpty)
        ? user.last_name![0].toUpperCase()
        : "";
    return "$first$last";
  }

  @override
  void initState() {
    super.initState();
    _loadImages();
    _loadExercises();
    _loadUser();

    _controller = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 2200),
    );

    _scale1 = Tween<double>(begin: 0.8, end: 1.0).animate(
      CurvedAnimation(
          parent: _controller,
          curve: const Interval(0.0, 0.3, curve: Curves.easeOutBack)),
    );
    _scale2 = Tween<double>(begin: 0.8, end: 1.0).animate(
      CurvedAnimation(
          parent: _controller,
          curve: const Interval(0.3, 0.6, curve: Curves.easeOutBack)),
    );
    _scale3 = Tween<double>(begin: 0.8, end: 1.0).animate(
      CurvedAnimation(
          parent: _controller,
          curve: const Interval(0.6, 0.9, curve: Curves.easeOutBack)),
    );

    _fade1 = Tween<double>(begin: 0.0, end: 1.0).animate(
      CurvedAnimation(parent: _controller, curve: const Interval(0.0, 0.3)),
    );
    _fade2 = Tween<double>(begin: 0.0, end: 1.0).animate(
      CurvedAnimation(parent: _controller, curve: const Interval(0.3, 0.6)),
    );
    _fade3 = Tween<double>(begin: 0.0, end: 1.0).animate(
      CurvedAnimation(parent: _controller, curve: const Interval(0.6, 0.9)),
    );

    _slide1 = Tween<Offset>(begin: const Offset(0, 0.4), end: Offset.zero)
        .animate(CurvedAnimation(
            parent: _controller,
            curve: const Interval(0.0, 0.3, curve: Curves.easeOutBack)));
    _slide2 = Tween<Offset>(begin: const Offset(0, 0.4), end: Offset.zero)
        .animate(CurvedAnimation(
            parent: _controller,
            curve: const Interval(0.3, 0.6, curve: Curves.easeOutBack)));
    _slide3 = Tween<Offset>(begin: const Offset(0, 0.4), end: Offset.zero)
        .animate(CurvedAnimation(
            parent: _controller,
            curve: const Interval(0.6, 0.9, curve: Curves.easeOutBack)));

    Future.delayed(const Duration(milliseconds: 300), () {
      if (mounted) _controller.forward();
    });
  }

  Future<void> _loadImages() async {
    try {
      final List<dynamic> data = await HomeService.getAllHomePages();
      setState(() {
        homeImages = {
          for (var item in data) item['id'] as int: item['image_url'] as String,
        };
        isLoading = false;
      });
    } catch (e) {
      debugPrint("Error loading homePage: $e");
      setState(() {
        isLoading = false;
      });
    }
  }

  @override
  Widget build(BuildContext context) {
    final t = AppLocalizations.of(context)!;

    return Scaffold(
      backgroundColor: Theme.of(context).scaffoldBackgroundColor,
      body: Stack(
        children: [
          SafeArea(
            child: ListView(
              padding: const EdgeInsets.symmetric(horizontal: 16, vertical: 12),
              children: [
                // Header
                // Row(
                //   mainAxisAlignment: MainAxisAlignment.spaceBetween,
                //   children: [
                //     Text(
                //       t.welcome + ' '+(user?.first_name ?? ""),
                //       style: Theme.of(context).textTheme.displayLarge,
                //     ),

                // TEST TEST
                Row(
                  mainAxisAlignment: MainAxisAlignment.spaceBetween,
                  children: [
                    // Welcome + اسم المستخدم
                    LayoutBuilder(
                      builder: (context, constraints) {
                        final screenWidth = MediaQuery.of(context).size.width;
                        final fontSize = screenWidth * 0.05; // 5% من عرض الشاشة
                        return Text(
                          t.welcome + ' ' + (user?.first_name ?? ""),
                          style: TextStyle(
                            fontSize: fontSize.clamp(16, 24), // بين 16 و 24px
                            fontWeight: FontWeight.bold,
                            color: Theme.of(context).textTheme.bodyLarge!.color,
                          ),
                        );
                      },
                    ),
                    // TEST TEST ^^^^^
                    Row(
                      children: [
                        GestureDetector(
                       //   onTap: () {
                        //    Navigator.push(
                         //     context,
                          //    MaterialPageRoute(
                          //      builder: (_) => const EditProfileScreen(),
                          //    ),
                         //   );
                         // },
                          child: CircleAvatar(
                            radius: 17,
                            backgroundColor: Colors.blueAccent,
                            child: Text(
                              getInitials(user),
                              style: const TextStyle(
                                  fontSize: 15, color: Colors.white),
                            ),
                          ),
                        ),
                        IconButton(
                          icon: Icon(Icons.settings,
                              color: Theme.of(context).iconTheme.color),
                          onPressed: () {
                            Navigator.push(
                              context,
                              MaterialPageRoute(
                                  builder: (_) => const SettingsScreen()),
                            );
                          },
                        ),
                      ],
                    ),

                  ],
                ),
                const SizedBox(height: 24),

                // Animated Cards
                FadeTransition(
                  opacity: _fade1,
                  child: SlideTransition(
                    position: _slide1,
                    child: ScaleTransition(
                      scale: _scale1,
                      child: Row(
                        mainAxisAlignment: MainAxisAlignment.spaceEvenly,
                        children: [
                          Expanded(
                            child: _buildCard(
                              title: t.exercises,
                              imageUrl: homeImages[1] != null
                                  ? ApiConstants.baseUrl +
                                      '/uploads/images/' +
                                      homeImages[1]!
                                  : 'assets/images/logo_landingpage_NoBackground.png',
                              color: Colors.white,
                              height: 240,
                              onTap: () {
                                mainTabNotifier.value = 2;
                              },
                            ),
                          ),
                          const SizedBox(width: 10),
                          Expanded(
                            child: _buildCard(
                              title: t.matches,
                              imageUrl: homeImages[2] != null
                                  ? ApiConstants.baseUrl +
                                      '/uploads/images/' +
                                      homeImages[2]!
                                  : 'assets/images/logo_landingpage_NoBackground.png',
                              color: Colors.white,
                              height: 240,
                              onTap: () {
                                Navigator.push(
                                  context,
                                  MaterialPageRoute(
                                      builder: (_) => const MatchesScreen()),
                                );
                              },
                            ),
                          ),
                        ],
                      ),
                    ),
                  ),
                ),

                const SizedBox(height: 16),

                FadeTransition(
                  opacity: _fade2,
                  child: SlideTransition(
                    position: _slide2,
                    child: ScaleTransition(
                      scale: _scale2,
                      child: _buildCard(
                        title: t.live_stream,
                        imageUrl: homeImages[3] != null
                            ? ApiConstants.baseUrl +
                                '/uploads/images/' +
                                homeImages[3]!
                            : 'assets/images/logo_landingpage_NoBackground.png',
                        color: Colors.white,
                        height: 180,
                        onTap: () {
                          mainTabNotifier.value = 3;                        },
                      ),
                    ),
                  ),
                ),

                const SizedBox(height: 24),

                // Ads Section
                Padding(
                  padding: const EdgeInsets.symmetric(horizontal: 16),
                  child: Text(t.latest_announcements,
                      style: Theme.of(context).textTheme.titleLarge),
                ),
                const SizedBox(height: 12),
                const AdsHorizontalList(),

                const SizedBox(height: 24),
                _buildCarouselWithExercises(
                  t.trainings,
                  exercises,
                  (exercise) {
                    Navigator.push(
                      context,
                      MaterialPageRoute(
                          builder: (_) =>
                              ExerciseDetailScreen(exercise: exercise)),
                    );
                  },
                ),
                const SizedBox(height: 24),

                // About Us Button
                Center(
                  child: ElevatedButton(
                    onPressed: () {
                      Navigator.push(
                        context,
                        MaterialPageRoute(builder: (_) => const AboutUsPage()),
                      );
                    },
                    style: ElevatedButton.styleFrom(
                      backgroundColor: Colors.deepPurple,
                      padding: const EdgeInsets.symmetric(vertical: 14),
                      minimumSize: const Size(double.infinity, 60),
                      shape: RoundedRectangleBorder(
                        borderRadius: BorderRadius.circular(12),
                      ),
                    ),
                    child: Text(
                      t.about_us,
                      style: const TextStyle(
                          fontSize: 20, fontWeight: FontWeight.bold),
                    ),
                  ),
                ),
              ],
            ),
          ),

          // Loader Overlay
          if (isLoading)
            Container(
              color: Colors.black.withOpacity(0.4),
              child: const Center(
                child: CircularProgressIndicator(),
              ),
            ),
        ],
      ),
    );
  }

  Widget _buildCard({
    required String title,
    required String imageUrl,
    required Color color,
    Color? textColor,
    double? height,
    VoidCallback? onTap,
  }) {
    Widget cardContent = Container(
      height: height ?? 140,
      alignment: Localizations.localeOf(context).languageCode == 'ar'
    ? Alignment.bottomRight
    : Alignment.bottomLeft,
      decoration: BoxDecoration(
        color: color.withOpacity(0.1),
        borderRadius: BorderRadius.circular(15),
        image: imageUrl.isNotEmpty
            ? DecorationImage(
                image: imageUrl.startsWith("http")
                    ? NetworkImage(imageUrl)
                    : AssetImage(imageUrl) as ImageProvider,
                fit: BoxFit.cover,
                colorFilter:
                    ColorFilter.mode(color.withOpacity(0.5), BlendMode.darken),
              )
            : null,
      ),
      padding: const EdgeInsets.all(16),
      child: Text(
        title,
        textAlign: Localizations.localeOf(context).languageCode == 'ar'
            ? TextAlign.right
            : TextAlign.left,
        style: TextStyle(
          color: textColor ?? Colors.white,
          fontSize: 20,
          fontWeight: FontWeight.bold,
          shadows: const [
            Shadow(color: Colors.black54, blurRadius: 4, offset: Offset(1, 1)),
          ],
        ),
      ),
    );

    if (onTap != null) {
      return GestureDetector(
        onTap: onTap,
        child: cardContent,
      );
    } else {
      return cardContent;
    }
  }

  Widget _buildCarouselWithExercises(
      String title, List<Exercise> items, Function(Exercise) onTap) {
    final locale = Localizations.localeOf(context).languageCode;

    return Padding(
      padding: const EdgeInsets.symmetric(vertical: 16),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Padding(
            padding: const EdgeInsets.symmetric(horizontal: 16),
            child: Text(
              title,
              style: const TextStyle(
                fontSize: 20,
                fontWeight: FontWeight.bold,
                color: Colors.white,
              ),
            ),
          ),
          const SizedBox(height: 8),
          CarouselSlider(
            options: CarouselOptions(
              height: 180,
              enlargeCenterPage: true,
              enableInfiniteScroll: true,
              autoPlay: true,
              autoPlayInterval: const Duration(seconds: 3),
            ),
            items: items.map((exercise) {
              final imageUrl =
                  ApiConstants.baseUrl + '/uploads/images/' + exercise.image;

              // اختيار العنوان والوصف حسب اللغة
              final titleText = (locale == "ar")
                  ? (exercise.title_ar ?? "")
                  : (exercise.title_en ?? "");

              return GestureDetector(
                onTap: () => onTap(exercise),
                child: Container(
                  margin: const EdgeInsets.symmetric(horizontal: 8),
                  decoration: BoxDecoration(
                    borderRadius: BorderRadius.circular(16),
                    image: DecorationImage(
                      image: NetworkImage(imageUrl),
                      fit: BoxFit.cover,
                    ),
                  ),
                  alignment: locale == "ar"
                      ? Alignment.bottomRight
                      : Alignment.bottomLeft,
                  padding: const EdgeInsets.all(12),
                  child: Text(
                    titleText.length > 10
                        ? titleText.substring(0, 10) + '...'
                        : titleText,
                    style: const TextStyle(
                      color: Colors.white,
                      fontSize: 14,
                      fontWeight: FontWeight.bold,
                      shadows: [
                        Shadow(
                          blurRadius: 4,
                          color: Colors.black45,
                          offset: Offset(1, 1),
                        ),
                      ],
                    ),
                  ),
                ),
              );
            }).toList(),
          ),
        ],
      ),
    );
  }
}
