import 'package:flutter/material.dart';
import 'package:hockey_app/models/category_model.dart';
import 'package:hockey_app/services/api_constants.dart';
import 'package:provider/provider.dart';
import '../models/exercise_model.dart';
import '../providers/locale_provider.dart';
import '../services/exercise_service.dart';
import '../services/category_service.dart';
import 'exercise_detail.dart';
import '../l10n/app_localizations.dart';

class ExerciseScreen extends StatefulWidget {
  const ExerciseScreen({Key? key}) : super(key: key);

  @override
  State<ExerciseScreen> createState() => _ExerciseScreenState();
}

class _ExerciseScreenState extends State<ExerciseScreen> {
  int? selectedCategoryId; // null = All
  late Future<List<Exercise>> _exercisesFuture;
  late Future<List<Category>> _categoriesFuture;

  @override
  void initState() {
    super.initState();
    _exercisesFuture = ExerciseService.getExercises();
    _categoriesFuture = CategoryService.getCategory();
  }

  @override
  Widget build(BuildContext context) {
    final tr = AppLocalizations.of(context)!;
    final locale =
        Provider.of<LocaleProvider>(context, listen: false).locale.languageCode;

    return Scaffold(
      backgroundColor: Theme.of(context).scaffoldBackgroundColor,
      appBar: AppBar(
        title: Text(tr.exercises),
        backgroundColor: Theme.of(context).appBarTheme.backgroundColor,
        foregroundColor: Theme.of(context).appBarTheme.foregroundColor,
        elevation: Theme.of(context).appBarTheme.elevation,
      ),
      body: Column(
        children: [
          FutureBuilder<List<Category>>(
            future: _categoriesFuture,
            builder: (context, snapshot) {
              if (snapshot.connectionState == ConnectionState.waiting) {
                return const LinearProgressIndicator();
              }
              if (snapshot.hasError) {
                return Text(tr.error_loading_categories);
              }
              final categories = snapshot.data ?? [];
              return _buildFilterBar(categories, tr, locale);
            },
          ),
          Expanded(
            child: FutureBuilder<List<Exercise>>(
              future: _exercisesFuture,
              builder: (context, snapshot) {
                if (snapshot.connectionState == ConnectionState.waiting) {
                  return const Center(child: CircularProgressIndicator());
                }
                if (snapshot.hasError) {
                  return Center(child: Text(tr.error_loading_exercises));
                }
                final exercises = snapshot.data ?? [];
                final filtered = selectedCategoryId == null
                    ? exercises
                    : exercises
                    .where((e) => e.category == selectedCategoryId)
                    .toList();

                return Padding(
                  padding: const EdgeInsets.all(12),
                  child: GridView.count(
                    crossAxisCount: 2,
                    crossAxisSpacing: 12,
                    mainAxisSpacing: 12,
                    childAspectRatio: MediaQuery.of(context).orientation == Orientation.portrait
                        ? 3 / 4
                        : 4 / 3,
                    children: filtered.map((e) => _buildExerciseCard(e, locale)).toList(),
                  ),
                );
              },
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildFilterBar(
      List<Category> categories, AppLocalizations tr, String locale) {
    return Container(
      height: 50,
      margin: const EdgeInsets.symmetric(horizontal: 12, vertical: 8),
      child: ListView(
        scrollDirection: Axis.horizontal,
        children: [
          Padding(
            padding: const EdgeInsets.only(right: 8),
            child: ChoiceChip(
              label: Text(tr.all),
              selected: selectedCategoryId == null,
              selectedColor: Theme.of(context).chipTheme.selectedColor,
              backgroundColor: Theme.of(context).chipTheme.backgroundColor,
              onSelected: (_) {
                setState(() {
                  selectedCategoryId = null;
                });
              },
            ),
          ),
          ...categories.map((cat) {
            final isSelected = selectedCategoryId == cat.id;
            final catTitle =
            (locale == "ar") ? (cat.title_ar ?? "") : (cat.title_en ?? "");
            return Padding(
              padding: const EdgeInsets.only(right: 8),
              child: ChoiceChip(
                label: Text(catTitle),
                selected: isSelected,
                selectedColor: Theme.of(context).chipTheme.selectedColor,
                backgroundColor: Theme.of(context).chipTheme.backgroundColor,
                onSelected: (_) {
                  setState(() {
                    selectedCategoryId = cat.id;
                  });
                },
              ),
            );
          }).toList(),
        ],
      ),
    );
  }

  Widget _buildExerciseCard(Exercise exercise, String locale) {
    final title =
    (locale == "ar") ? (exercise.title_ar ?? "") : (exercise.title_en ?? "");
    final description = (locale == "ar")
        ? (exercise.description_ar ?? "")
        : (exercise.description_en ?? "");

    return GestureDetector(
      onTap: () {
        Navigator.push(
          context,
          MaterialPageRoute(
            builder: (_) => ExerciseDetailScreen(exercise: exercise),
          ),
        );
      },
      child: Container(
        decoration: BoxDecoration(
          color: Theme.of(context).cardColor,
          borderRadius: BorderRadius.circular(16),
          boxShadow: [
            BoxShadow(
              color: Colors.black.withOpacity(0.2),
              blurRadius: 6,
              offset: const Offset(0, 4),
            ),
          ],
        ),
        child: Column(
          crossAxisAlignment: CrossAxisAlignment.start,
          children: [
            Hero(
              tag: title, // Hero animation uses exercise title as tag
              child: ClipRRect(
                // ClipRRect ensures the image corners are clipped to the card's border radius
                borderRadius: const BorderRadius.vertical(top: Radius.circular(16)),
                child: Image.network(
                  ApiConstants.baseUrl + "/uploads/images/" + exercise.image,
                  height: 200,
                  width: double.infinity,
                  fit: BoxFit.cover, // Makes the image cover the available space
                  errorBuilder: (_, __, ___) => Image.asset(
                    "assets/images/exer1.jpeg",
                    fit: BoxFit.cover, // Fallback image if loading fails
                  ),
                ),
              ),
            ),

            Padding(
              padding: const EdgeInsets.all(10),
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  Text(
                    title,
                    style: Theme.of(context).textTheme.titleMedium?.copyWith(
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                  const SizedBox(height: 6),
                  // exercise Description in the exercise page
                  // Text(
                  //   description,
                  //   maxLines: 2,
                  //   overflow: TextOverflow.ellipsis,
                  //   style: Theme.of(context).textTheme.bodyMedium,
                  // ),
                ],
              ),
            ),
          ],
        ),
      ),
    );
  }
}
