import 'package:flutter/material.dart';
import 'package:provider/provider.dart';
import 'package:hockey_app/screens/start_work_out.dart';
import '../l10n/app_localizations.dart';
import '../models/exercise_model.dart';
import '../models/category_model.dart';
import '../services/category_service.dart';
import '../services/exercise_service.dart';
import '../services/api_constants.dart';
import '../providers/locale_provider.dart';

class ExerciseDetailScreen extends StatefulWidget {
  final Exercise exercise;

  const ExerciseDetailScreen({super.key, required this.exercise});

  @override
  State<ExerciseDetailScreen> createState() => _ExerciseDetailScreenState();
}

class _ExerciseDetailScreenState extends State<ExerciseDetailScreen> {
  String categoryName = "";
  late Future<List<Exercise>> _exerciseListFuture;

  @override
  void initState() {
    super.initState();
    _loadCategory();
    _exerciseListFuture = _loadExercises();
  }

  Future<void> _loadCategory() async {
    try {
      final categories = await CategoryService.getCategory();
      final cat = categories.firstWhere(
            (c) => c.id == widget.exercise.category,
        orElse: () => Category(
            id: 0, title_ar: "Unknown", title_en: "Unknown"), // fallback
      );

      final locale = Provider.of<LocaleProvider>(context, listen: false)
          .locale
          .languageCode;

      setState(() {
        categoryName =
        (locale == "ar") ? cat.title_ar ?? "Unknown" : cat.title_en ?? "Unknown";
      });
    } catch (e) {
      setState(() {
        categoryName = "Unknown";
      });
    }
  }

  Future<List<Exercise>> _loadExercises() async {
    try {
      final allExercises = await ExerciseService.getExercises();
      final filtered =
      allExercises.where((e) => e.id != widget.exercise.id).toList();
      return filtered;
    } catch (e) {
      return [];
    }
  }

  @override
  Widget build(BuildContext context) {
    final theme = Theme.of(context);
    final tr = AppLocalizations.of(context)!;
    final locale = Provider.of<LocaleProvider>(context).locale.languageCode;

    // النصوص للتمرين الحالي
    final title =
    (locale == "ar") ? widget.exercise.title_ar ?? "" : widget.exercise.title_en ?? "";
    final description = (locale == "ar")
        ? widget.exercise.description_ar ?? tr.description_placeholder
        : widget.exercise.description_en ?? tr.description_placeholder;

    return Scaffold(
      appBar: AppBar(
        title: Text(title),
        centerTitle: true,
      ),
      body: FutureBuilder<List<Exercise>>(
        future: _exerciseListFuture,
        builder: (context, snapshot) {
          if (snapshot.connectionState == ConnectionState.waiting) {
            return const Center(child: CircularProgressIndicator());
          }
          if (snapshot.hasError || snapshot.data!.isEmpty) {
            return Center(child: Text(tr.no_exercises_available));
          }
          final exercises = snapshot.data!;

          return ListView(
            padding: const EdgeInsets.only(bottom: 80),
            children: [
              // صورة التمرين الرئيسية
              Container(
                height: 250,
                width: double.infinity,
                decoration: BoxDecoration(
                  image: DecorationImage(
                    image: NetworkImage(
                        '${ApiConstants.baseUrl}/uploads/images/${widget.exercise.image ?? ""}'),
                    fit: BoxFit.cover,
                  ),
                ),
              ),
              const SizedBox(height: 16),

              // الكاتيجوري
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 16.0),
                child: Chip(
                  label: Text(categoryName),
                  backgroundColor: Colors.blue.shade100,
                ),
              ),
              const SizedBox(height: 12),

              // الوصف الكامل للتمرين
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 16.0),
                child: Text(
                  description,
                  style: theme.textTheme.bodyMedium,
                ),
              ),
              const SizedBox(height: 16),

              // التفاصيل
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 16.0),
                child: Row(
                  mainAxisAlignment: MainAxisAlignment.spaceAround,
                  children: [
                    _infoColumn(widget.exercise.level ?? tr.advanced, Icons.arrow_upward),
                    _infoColumn("${widget.exercise.time ?? ""} ${tr.time_unit}", Icons.timer),
                    _infoColumn("${widget.exercise.duration ?? ""} ${tr.numberUnit}", Icons.repeat),
                  ],
                ),
              ),
              const SizedBox(height: 24),

              // عنوان القائمة
              Padding(
                padding: const EdgeInsets.symmetric(horizontal: 16.0),
                child: Text(tr.exercise_list, style: theme.textTheme.titleLarge),
              ),
              const SizedBox(height: 12),

              // القائمة الديناميكية
              ...exercises.map((exercise) {
                final cardTitle =
                (locale == "ar") ? exercise.title_ar ?? "" : exercise.title_en ?? "";
                final cardDesc = (locale == "ar")
                    ? exercise.description_ar ?? tr.description_placeholder
                    : exercise.description_en ?? tr.description_placeholder;

                return Padding(
                  padding:
                  const EdgeInsets.symmetric(horizontal: 16.0, vertical: 6.0),
                  child: InkWell(
                    onTap: () {
                      Navigator.pushReplacement(
                        context,
                        MaterialPageRoute(
                          builder: (_) => ExerciseDetailScreen(exercise: exercise),
                        ),
                      );
                    },
                    child: Card(
                      child: Padding(
                        padding: const EdgeInsets.all(12),
                        child: Row(
                          children: [
                            // صورة صغيرة داخل الكرت
                            ClipRRect(
                              borderRadius: BorderRadius.circular(12),
                              child: Image.network(
                                '${ApiConstants.baseUrl}/uploads/images/${exercise.image ?? ""}',
                                height: 60,
                                width: 60,
                                fit: BoxFit.cover,
                                errorBuilder: (_, __, ___) => Image.asset(
                                  "assets/images/exer1.jpeg",
                                  height: 60,
                                  width: 60,
                                  fit: BoxFit.cover,
                                ),
                              ),
                            ),
                            const SizedBox(width: 12),
                            Expanded(
                              child: Column(
                                crossAxisAlignment: CrossAxisAlignment.start,
                                children: [
                                  Text(cardTitle, style: theme.textTheme.bodyMedium),
                                  const SizedBox(height: 4),

                                  // وصف مقصور على 10 أحرف
                                  Text(
                                    (cardDesc.length > 10)
                                        ? "${cardDesc.substring(0, 10)}..."
                                        : cardDesc,
                                    style: theme.textTheme.bodySmall,
                                  ),

                                  const SizedBox(height: 8),
                                  Row(
                                    children: [
                                      const Icon(Icons.fitness_center, size: 16),
                                      const SizedBox(width: 4),
                                      Flexible(
                                        child: Text(
                                          "${exercise.level ?? tr.advanced}",
                                          maxLines: 1,
                                          overflow: TextOverflow.ellipsis,
                                        ),
                                      ),
                                      const SizedBox(width: 12),
                                      const Icon(Icons.timer, size: 16),
                                      const SizedBox(width: 4),
                                      Flexible(
                                        child: Text(
                                          "${exercise.time ?? 30} ${tr.time_unit}",
                                          maxLines: 1,
                                          overflow: TextOverflow.ellipsis,
                                        ),
                                      ),
                                      const SizedBox(width: 12),
                                      const Icon(Icons.repeat, size: 16),
                                      const SizedBox(width: 4),
                                      Flexible(
                                        child: Text(
                                          "${exercise.duration ?? 8.25} ${tr.numberUnit}",
                                          maxLines: 1,
                                          overflow: TextOverflow.ellipsis,
                                        ),
                                      ),
                                    ],
                                  )
                                ],
                              ),
                            )
                          ],
                        ),
                      ),
                    ),
                  ),
                );
              }).toList(),
            ],
          );
        },
      ),
      bottomNavigationBar: SafeArea(
        minimum: const EdgeInsets.only(bottom: 12),
        child: Padding(
          padding: const EdgeInsets.symmetric(horizontal: 16.0),
          child: SizedBox(
            width: double.infinity,
            height: 56,
            child: ElevatedButton(
              onPressed: () {
                Navigator.push(
                  context,
                  MaterialPageRoute(
                    builder: (_) => StartWorkoutScreen(exercise: widget.exercise),
                  ),
                );
              },
              child: Text(tr.start_workout),
            ),
          ),
        ),
      ),
    );
  }

  static Column _infoColumn(String text, IconData icon) {
    return Column(
      children: [
        Icon(icon, size: 28),
        const SizedBox(height: 4),
        Text(text, style: const TextStyle(fontWeight: FontWeight.bold)),
      ],
    );
  }
}
