import 'dart:io';

import 'package:flutter/material.dart';
import 'package:hockey_app/services/api_constants.dart';
import '../models/user_model.dart';
import '../services/user_service.dart';
import 'package:image_picker/image_picker.dart';

class EditProfileScreen extends StatefulWidget {
  const EditProfileScreen({super.key});

  @override
  State<EditProfileScreen> createState() => _EditProfileScreenState();
}

class _EditProfileScreenState extends State<EditProfileScreen> {
  User? user;
  final TextEditingController _bioController = TextEditingController();
  bool isLoading = true;
  XFile? _pickedImage;

  @override
  void initState() {
    super.initState();
    _loadUser();
  }

  Future<void> _loadUser() async {
    user = await UserService.getUser();
    _bioController.text = user?.bio ?? "";
    setState(() {
      isLoading = false;
    });
  }

  Future<void> _pickImage() async {
    try {
      final picker = ImagePicker();
      final picked = await picker.pickImage(source: ImageSource.gallery);

      if (picked != null && user != null) {
        setState(() {
          _pickedImage = picked;
        });

        final file = File(picked.path);

        final success = await UserService.uploadProfileImage(user!.id!, file);

        if (success) {
          await _loadUser(); // إعادة تحميل بيانات المستخدم بعد التحديث
          ScaffoldMessenger.of(context).showSnackBar(
            const SnackBar(content: Text("Profile image updated successfully")),
          );
        } else {
          ScaffoldMessenger.of(context).showSnackBar(
            const SnackBar(content: Text("Failed to upload image")),
          );
        }
      }
    } catch (e) {
      debugPrint("Error picking/uploading image: $e");
      ScaffoldMessenger.of(context).showSnackBar(
        const SnackBar(content: Text("An error occurred")),
      );
    }
  }


  Future<void> _saveBio() async {
    final bio = _bioController.text.trim();
    if (bio.isEmpty) return;

    setState(() {
      isLoading = true;
    });
    await UserService.updateUserBio(user!.id as int ,bio);
    await _loadUser();
    ScaffoldMessenger.of(context).showSnackBar(
      const SnackBar(content: Text("Bio updated successfully")),
    );
    setState(() {
      isLoading = false;
    });
  }

  @override
  void dispose() {
    _bioController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      appBar: AppBar(
        title: const Text("Edit Profile"),
        centerTitle: true,
      ),
      body: isLoading
          ? const Center(child: CircularProgressIndicator())
          : Padding(
        padding: const EdgeInsets.all(16.0),
        child: Column(
          children: [
            // صورة المستخدم
            GestureDetector(
              onTap: _pickImage,
              child: CircleAvatar(
                radius: 50,
                backgroundImage: _pickedImage != null
                    ? FileImage(
                  File(_pickedImage!.path),
                ) as ImageProvider
                    : (user?.profileImageUrl != null
                    ? NetworkImage(ApiConstants.baseUrl+'/uploads/images/'+  user!.profileImageUrl!)
                    : const AssetImage("assets/images/user.png")
                as ImageProvider),
              ),
            ),
            const SizedBox(height: 24),

            TextField(
              controller: _bioController,
              maxLines: 5,
              textDirection: Directionality.of(context), // ياخد اتجاه التطبيق الحالي
              decoration: InputDecoration(
                hintText: "اكتب نبذة عنك...",
                border: OutlineInputBorder(
                  borderRadius: BorderRadius.circular(12),
                ),
              ),
            ),

            const SizedBox(height: 24),

            SizedBox(
              width: double.infinity,
              height: 50,
              child: ElevatedButton(
                onPressed: _saveBio,
                child: const Text("حفظ"),
              ),
            ),
          ],
        ),
      ),
    );
  }
}
