import 'package:flutter/material.dart';
import '../observers/route_observer.dart';
import '../services/api_constants.dart';
import 'package:http/http.dart' as http;
import 'dart:convert';
import '../services/news_service.dart';
import '../l10n/app_localizations.dart';

class AdsHorizontalList extends StatefulWidget {
  const AdsHorizontalList({super.key});

  @override
  State<AdsHorizontalList> createState() => _AdsHorizontalListState();
}

class _AdsHorizontalListState extends State<AdsHorizontalList>
    with TickerProviderStateMixin, RouteAware {
  late List<AnimationController> _controllers;
  late List<Animation<double>> _scaleAnimations;
  late List<Animation<double>> _fadeAnimations;

  List<Map<String, dynamic>> ads = [];
  bool isLoading = true;

  @override
  void initState() {
    super.initState();
    _fetchAds();
  }

  Future<void> _fetchAds() async {
    try {
      final data = await UpdatesService.getAnnouncements();
      setState(() {
        ads = data;
        isLoading = false;
      });
      _initAnimations();
    } catch (e) {
      print('Error fetching ads: $e');
      setState(() {
        isLoading = false;
      });
    }
  }

  void _initAnimations() {
    _controllers = List.generate(
      ads.length,
      (index) => AnimationController(
        vsync: this,
        duration: const Duration(milliseconds: 700),
      ),
    );

    _scaleAnimations = _controllers
        .map((controller) => Tween<double>(begin: 0.5, end: 1.0).animate(
              CurvedAnimation(parent: controller, curve: Curves.easeOutBack),
            ))
        .toList();

    _fadeAnimations = _controllers
        .map((controller) => Tween<double>(begin: 0.0, end: 1.0).animate(
              CurvedAnimation(parent: controller, curve: Curves.easeIn),
            ))
        .toList();

    _startAnimations();
  }

  Future<void> _startAnimations() async {
    for (var i = 0; i < _controllers.length; i++) {
      _controllers[i].reset();
      await Future.delayed(const Duration(milliseconds: 200));
      if (mounted) _controllers[i].forward();
    }
  }

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();
    routeObserver.subscribe(this, ModalRoute.of(context)!);
  }

  @override
  void dispose() {
    for (var c in _controllers) {
      c.dispose();
    }
    routeObserver.unsubscribe(this);
    super.dispose();
  }

  @override
  void didPopNext() {
    _startAnimations();
  }

  @override
  Widget build(BuildContext context) {
    final tr = AppLocalizations.of(context)!;

    if (isLoading) {
      return const SizedBox(
        height: 140,
        child: Center(child: CircularProgressIndicator()),
      );
    }

    return SizedBox(
      height: 140,
      child: ListView.separated(
        scrollDirection: Axis.horizontal,
        itemCount: ads.length,
        separatorBuilder: (_, __) => const SizedBox(width: 12),
        itemBuilder: (context, index) {
          final ad = ads[index];
          final imageUrl =
              '${ApiConstants.baseUrl}/uploads/images/${ad['image']}';

          String description = ad['description'] ?? '';
          if (description.length > 10) {
            description = description.substring(0, 10) + '...';
          }

          return FadeTransition(
            opacity: _fadeAnimations[index],
            child: ScaleTransition(
              scale: _scaleAnimations[index],
              child: GestureDetector(
                onTap: () {
                  print('${tr.clicked_ad}: ${ad['title']}');
                },
                child: Container(
                  width: 280,
                  height: 140,
                  decoration: BoxDecoration(
                    borderRadius: BorderRadius.circular(16),
                    boxShadow: [
                      BoxShadow(
                        color: Colors.black.withOpacity(0.15),
                        blurRadius: 6,
                        offset: const Offset(2, 4),
                      ),
                    ],
                  ),
                  child: ClipRRect(
                    borderRadius: BorderRadius.circular(16),
                    child: Stack(
                      fit: StackFit.expand,
                      children: [
                        Image.network(
                          imageUrl,
                          fit: BoxFit.cover,
                        ),
                        Container(
                          color: Colors.black.withOpacity(0.3),
                        ),
                        Padding(
                          padding: const EdgeInsets.all(12.0),
                          child: Align(
                            alignment: Alignment.bottomLeft,
                            child: Text(
                              description,
                              style: const TextStyle(
                                color: Colors.white,
                                fontSize: 14,
                                fontWeight: FontWeight.bold,
                                shadows: [
                                  Shadow(
                                    blurRadius: 4,
                                    color: Colors.black45,
                                    offset: Offset(1, 1),
                                  ),
                                ],
                              ),
                            ),
                          ),
                        ),
                      ],
                    ),
                  ),
                ),
              ),
            ),
          );
        },
      ),
    );
  }
}
