import 'dart:convert';

import 'package:flutter/material.dart';
import 'package:hockey_app/screens/content_detail_screen.dart';
import 'package:hockey_app/screens/exercise_screen.dart';
import 'package:hockey_app/screens/livestream_screen.dart';
import 'package:hockey_app/screens/newAdsScreen.dart';
import 'package:hockey_app/screens/uniform_screen.dart';
import 'package:hockey_app/services/api_constants.dart';
import 'package:hockey_app/services/auth_service.dart';
import 'package:provider/provider.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'package:flutter_localizations/flutter_localizations.dart';
import 'package:firebase_core/firebase_core.dart';
import 'package:firebase_messaging/firebase_messaging.dart';

import 'firebase_options.dart';
import 'services/notification_service.dart';
import 'theme/app_theme.dart';
import 'providers/theme_provider.dart';
import 'providers/locale_provider.dart';
import 'l10n/app_localizations.dart';
import 'screens/SplashPage.dart';
import 'screens/login.dart';
import 'screens/onboarding_screen.dart';
import 'screens/home_screen.dart';
import 'screens/match_detail.dart';

final GlobalKey<NavigatorState> navigatorKey = GlobalKey<NavigatorState>();
final ValueNotifier<int> mainTabNotifier = ValueNotifier<int>(0);

Future<void> _firebaseMessagingBackgroundHandler(RemoteMessage message) async {
  await Firebase.initializeApp(options: DefaultFirebaseOptions.currentPlatform);
  print('Handling a background message: ${message.messageId}');
}

void main() async {
  WidgetsFlutterBinding.ensureInitialized();

  await Firebase.initializeApp(options: DefaultFirebaseOptions.currentPlatform);

  // Notification Service
  await NotificationService.init();
  FirebaseMessaging.instance.subscribeToTopic('allUsers');
  FirebaseMessaging.onBackgroundMessage(_firebaseMessagingBackgroundHandler);

  // Theme & Locale
  final themeProvider = ThemeProvider();
  await themeProvider.loadTheme();

  final localeProvider = LocaleProvider();
  await localeProvider.loadLocale();

  final prefs = await SharedPreferences.getInstance();
  final token = prefs.getString("auth_token");
  bool isLoggedIn = false;

  if (token != null) {
    try {
      final active = await AuthService.checkUserActive(token);
      if (active) {
        isLoggedIn = true;
      } else {
        await prefs.remove("auth_token");
      }
    } catch (e) {
      print("Error checking user active status: $e");
      isLoggedIn = false;
    }
  }

  FirebaseMessaging.instance.getInitialMessage().then((RemoteMessage? message) {
    if (message != null) {
      _handleMessage(message.data);
    }
  });

  FirebaseMessaging.onMessageOpenedApp.listen((RemoteMessage message) {
    _handleMessage(message.data);
  });

  NotificationService.onNotificationClick = (data) {
    _handleMessage(data);
  };

  runApp(
    MultiProvider(
      providers: [
        ChangeNotifierProvider.value(value: themeProvider),
        ChangeNotifierProvider.value(value: localeProvider),
      ],
      child: HockeyApp(isLoggedIn: isLoggedIn),
    ),
  );
}

void _handleMessage(Map<String, dynamic> data) {
  final prettyData = const JsonEncoder.withIndent('  ').convert(data);
  print('📩 Notification Data:\n$prettyData');

  final type = data['type'];

  if (navigatorKey.currentState != null) {
    if (type == 'news' || type == 'ads') {
      navigatorKey.currentState!.push(
        MaterialPageRoute(
          builder: (_) => AdvDetailScreen(
            title: data['title'] ?? '',
            description: data['body'] ?? '',
            image: data['image'] ?? '',
          ),
        ),
      );
    } else if (type == 'live') {
      mainTabNotifier.value = 3;

      if (navigatorKey.currentState!.canPop() == false) {
        navigatorKey.currentState!.pushReplacement(
          MaterialPageRoute(builder: (_) => const MainScreen(initialIndex: 3)),
        );
      }
    }
  }
}

class HockeyApp extends StatelessWidget {
  final bool isLoggedIn;
  const HockeyApp({super.key, required this.isLoggedIn});

  @override
  Widget build(BuildContext context) {
    final themeProvider = Provider.of<ThemeProvider>(context);
    final localeProvider = Provider.of<LocaleProvider>(context);

    // FCM foreground listener
    FirebaseMessaging.onMessage.listen((RemoteMessage message) {
      String? imageUrl;

      if (message.notification?.android?.imageUrl != null) {
        imageUrl = ApiConstants.baseUrl+'/uploads/images/'+message.notification!.android!.imageUrl.toString();
      }
      else if (message.notification?.apple?.imageUrl != null) {
        imageUrl = ApiConstants.baseUrl+'/uploads/images/'+message.notification!.apple!.imageUrl.toString();
      }
      else if (message.data['image'] != null) {
        imageUrl = ApiConstants.baseUrl+'/uploads/images/'+message.data['image'];
      }

      print("Image URL: $imageUrl");

      NotificationService.showNotification(
        message.notification?.title ?? '',
        message.notification?.body ?? '',
        imageUrl:imageUrl ?? '',
        payload: message.data,
      );
    });

    FirebaseMessaging.instance.getToken().then((token) {
      print('FCM Token: $token');
    });

    return MaterialApp(
      navigatorKey: navigatorKey,
      title: 'Hockey App',
      debugShowCheckedModeBanner: false,
      themeMode: themeProvider.themeMode,
      theme: AppTheme.lightTheme,
      darkTheme: AppTheme.darkTheme,
      locale: localeProvider.locale,
      supportedLocales: const [Locale('en'), Locale('ar')],
      localizationsDelegates: const [
        AppLocalizations.delegate,
        GlobalMaterialLocalizations.delegate,
        GlobalWidgetsLocalizations.delegate,
        GlobalCupertinoLocalizations.delegate,
      ],
      builder: (context, child) {
        return Directionality(
          textDirection: localeProvider.locale.languageCode == 'ar'
              ? TextDirection.rtl
              : TextDirection.ltr,
          child: child!,
        );
      },
      home: SplashScreen(isLoggedIn: isLoggedIn),
      routes: {
        '/home': (_) => const OnboardingScreen(),
        '/login': (_) => const LoginPage(),
        '/main':(_)=> const MainScreen()
      },
    );
  }
}

class MainScreen extends StatefulWidget {
  final int initialIndex;
  const MainScreen({super.key, this.initialIndex = 0});

  @override
  State<MainScreen> createState() => _MainScreenState();
}

class _MainScreenState extends State<MainScreen> {
  late int _selectedIndex;
  late final List<Widget> _screens;

  @override
  void initState() {
    super.initState();
    _selectedIndex = widget.initialIndex;

    _screens = [
      HomeScreen(),
      UniformScreen(),
      ExerciseScreen(),
      LiveStreamScreen(),
      AdsNewsScreen(),
    ];

    mainTabNotifier.addListener(() {
      if (mounted) {
        setState(() => _selectedIndex = mainTabNotifier.value);
      }
    });
  }

  @override
  void dispose() {
    mainTabNotifier.removeListener(() {});
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final tr = AppLocalizations.of(context)!;

    return Scaffold(
      body: _screens[_selectedIndex],
      bottomNavigationBar: BottomNavigationBar(
        currentIndex: _selectedIndex,
        onTap: (index) {
          setState(() => _selectedIndex = index);
          mainTabNotifier.value = index;
        },
        showSelectedLabels: true,
        showUnselectedLabels: true,
        selectedItemColor: Colors.deepPurple,   // لون الأيقونة المحددة
        unselectedItemColor: Colors.grey,       // لون الأيقونات غير المحددة
        type: BottomNavigationBarType.fixed,
        items: [
          BottomNavigationBarItem(
            icon: const Icon(Icons.home),
            label: tr.home,
          ),
          BottomNavigationBarItem(icon: Image.asset('assets/images/uniform-lcon.png', width: 33, height: 33, color: Colors.grey,),
            activeIcon: Image.asset('assets/images/uniform-lcon.png', width: 33, height: 33, color: Colors.deepPurple,), label: tr.uniform,),
          BottomNavigationBarItem(icon: const Icon(Icons.fitness_center), label: tr.exercise,),
          BottomNavigationBarItem(icon: const Icon(Icons.live_tv), label: tr.live_stream,),
          BottomNavigationBarItem(icon: const Icon(Icons.campaign), label: tr.announcements,
          ),
        ],
      ),
    );
  }
}
